const express = require('express');
const router = express.Router();
const accountantController = require('../controllers/accountantController');
const authMiddleware = require('../middleware/auth');

// Financial summary
router.get('/summary', authMiddleware, accountantController.getFinancialSummary);

// Transactions
router.get('/transactions', authMiddleware, accountantController.getRecentTransactions);
router.post('/transactions', authMiddleware, accountantController.addTransaction);
router.delete('/transactions/:transactionId', authMiddleware, accountantController.deleteTransaction);

// Petty cash
router.get('/petty-cash', authMiddleware, accountantController.getPettyCash);
router.post('/petty-cash', authMiddleware, accountantController.addPettyCash);
router.post('/petty-cash/allocate', authMiddleware, accountantController.allocatePettyCash);
router.post('/petty-cash/top-up', authMiddleware, accountantController.topUpPettyCash);
router.delete('/petty-cash/:transactionId', authMiddleware, accountantController.deletePettyCashTransaction);

// Add funds to account
router.post('/add-funds', authMiddleware, accountantController.addFunds);

// Disburse loan
router.post('/disburse/:loanId', authMiddleware, accountantController.disburseLoan);

// Generate report
router.get('/report', authMiddleware, accountantController.generateReport);

// Generate PDF report
router.get('/report/pdf', authMiddleware, accountantController.generatePDFReport);

// Notifications
router.get('/notifications', authMiddleware, accountantController.getNotifications);
router.delete('/notifications/clear-all', authMiddleware, accountantController.clearAllNotifications);

// Search client loans for payment
router.get('/search-loans', authMiddleware, accountantController.searchClientLoans);

// Record payment
router.post('/record-payment', authMiddleware, accountantController.recordPayment);

// New accounting system routes
router.get('/account-balances', authMiddleware, accountantController.getAccountBalances);
router.get('/approved-loans', authMiddleware, accountantController.getApprovedLoans);
router.post('/disburse-with-accounting', authMiddleware, accountantController.disburseLoanWithAccounting);
router.post('/record-payment-with-accounting', authMiddleware, accountantController.recordPaymentWithAccounting);

// Payment management routes
router.get('/loan-payments/:loanId', authMiddleware, accountantController.getLoanPayments);
router.put('/payment/:paymentId', authMiddleware, accountantController.editPayment);
router.delete('/payment/:paymentId', authMiddleware, accountantController.deletePayment);

// Alerts and follow-ups
router.get('/check-overdue-loans', authMiddleware, accountantController.checkOverdueLoans);

// Interest tracking
router.get('/interest-earned', authMiddleware, accountantController.getInterestEarned);

// Account allocations (Cash, Bank, Mobile Money)
router.get('/account-allocations', authMiddleware, accountantController.getAccountAllocations);
router.post('/account-allocations/allocate', authMiddleware, accountantController.allocateAccountFunds);
router.post('/account-allocations/top-up', authMiddleware, accountantController.topUpAccount);

// Application fee confirmation
router.get('/pending-application-fees', authMiddleware, accountantController.getPendingApplicationFees);
router.post('/confirm-application-fee', authMiddleware, accountantController.confirmApplicationFee);

// Loan fees management
router.get('/disbursement-amount/:loan_id', authMiddleware, accountantController.calculateDisbursementAmount);

// Salary management
router.get('/staff-list', authMiddleware, accountantController.getStaffList);
router.get('/salary-history', authMiddleware, accountantController.getSalaryHistory);
router.post('/pay-salary', authMiddleware, accountantController.paySalary);

// Receipt management
router.get('/receipt/search/:receiptNumber', authMiddleware, accountantController.searchPaymentByReceipt);
router.get('/receipt/print/:receiptNumber', authMiddleware, accountantController.generatePaymentReceipt);

module.exports = router;