// SPFI Loan Management System - Frontend Application
// API Base URL
const API_BASE = '/api';

// Global state
let currentUser = null;
let authToken = null;

// Initialize app on load
document.addEventListener('DOMContentLoaded', () => {
    checkAuth();
    setupEventListeners();
    setupMobileMenu();
});

// Setup mobile menu toggle
function setupMobileMenu() {
    // Add click handler to navbar hamburger icon
    const navbar = document.querySelector('.navbar');
    if (navbar) {
        navbar.addEventListener('click', (e) => {
            // Check if click is on the hamburger area (left side)
            const rect = navbar.getBoundingClientRect();
            if (e.clientX - rect.left < 50) {
                toggleMobileMenu();
            }
        });
    }

    // Close menu when clicking on sidebar links
    document.querySelectorAll('.sidebar-link').forEach(link => {
        link.addEventListener('click', () => {
            if (window.innerWidth <= 768) {
                closeMobileMenu();
            }
        });
    });

    // Close menu when clicking outside
    document.addEventListener('click', (e) => {
        if (window.innerWidth <= 768) {
            const sidebar = document.querySelector('.sidebar');
            const navbar = document.querySelector('.navbar');
            if (sidebar && !sidebar.contains(e.target) && !navbar.contains(e.target)) {
                closeMobileMenu();
            }
        }
    });
}

function toggleMobileMenu() {
    const sidebar = document.querySelector('.sidebar');
    if (sidebar) {
        sidebar.classList.toggle('active');
    }
}

function closeMobileMenu() {
    const sidebar = document.querySelector('.sidebar');
    if (sidebar) {
        sidebar.classList.remove('active');
    }
}


// Check if user is already authenticated
function checkAuth() {
    const token = localStorage.getItem('authToken');
    const user = localStorage.getItem('currentUser');
    
    if (token && user) {
        authToken = token;
        currentUser = JSON.parse(user);
        showDashboard(currentUser.role);
    } else {
        showScreen('loginScreen');
    }
}

// Setup all event listeners
function setupEventListeners() {
    // Login form
    const loginForm = document.getElementById('loginForm');
    if (loginForm) {
        loginForm.addEventListener('submit', handleLogin);
    }

    // Modal forms
    // setupModalForms();
}

// Handle login
async function handleLogin(e) {
    e.preventDefault();
    
    const email = document.getElementById('email').value;
    const password = document.getElementById('password').value;
    const errorDiv = document.getElementById('loginError');
    
    console.log('Login attempt with email:', email);
    
    errorDiv.textContent = ''; // Clear previous errors
    
    try {
        const response = await apiCall('/auth/login', 'POST', { email, password });
        
        console.log('Login response:', response);
        
        if (response.success) {
            authToken = response.token;
            currentUser = response.user;
            
            // Store in localStorage
            localStorage.setItem('authToken', authToken);
            localStorage.setItem('currentUser', JSON.stringify(currentUser));
            
            console.log('Login successful, showing dashboard for role:', currentUser.role);
            
            // Clear URL parameters for security (remove credentials from URL)
            if (window.history && window.history.replaceState) {
                window.history.replaceState({}, document.title, window.location.pathname);
            }
            
            // Show appropriate dashboard
            showDashboard(currentUser.role);
        } else {
            const errorMessage = response.error || response.message || 'Login failed';
            console.error('Login failed:', errorMessage);
            errorDiv.textContent = errorMessage;
            errorDiv.style.display = 'block';
        }
    } catch (error) {
        console.error('Login error caught:', error);
        errorDiv.textContent = 'An error occurred. Please try again. Check console for details.';
        errorDiv.style.display = 'block';
    }
}

// Logout
function logout() {
    localStorage.removeItem('authToken');
    localStorage.removeItem('currentUser');
    authToken = null;
    currentUser = null;
    showScreen('loginScreen');
}

// API call helper
async function apiCall(endpoint, method = 'GET', body = null) {
    const options = {
        method,
        headers: {
            'Content-Type': 'application/json',
        }
    };
    
    if (authToken) {
        options.headers['Authorization'] = `Bearer ${authToken}`;
    }
    
    if (body && method !== 'GET') {
        options.body = JSON.stringify(body);
    }
    
    try {
        const response = await fetch(`${API_BASE}${endpoint}`, options);
        
        // Check if response is ok and content-type is JSON
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const text = await response.text();
            console.error('Non-JSON response:', text);
            throw new Error('Server returned non-JSON response');
        }
        
        const data = await response.json();
        return data;
    } catch (error) {
        console.error('API Error:', error);
        throw error;
    }
}

// Show specific screen
function showScreen(screenId) {
    document.querySelectorAll('.screen').forEach(screen => {
        screen.classList.remove('active');
    });
    document.getElementById(screenId).classList.add('active');
}

// Show dashboard based on role
function showDashboard(role) {
    console.log('showDashboard called with role:', role);
    
    // Hide login screen first
    document.getElementById('loginScreen').classList.remove('active');
    
    switch(role) {
        case 'manager':
            console.log('Showing manager dashboard');
            showScreen('managerDashboard');
            document.getElementById('managerName').textContent = currentUser.name;
            console.log('Screen shown, loading dashboard data...');
            loadManagerDashboard();
            break;
        case 'officer':
            showScreen('officerDashboard');
            document.getElementById('officerName').textContent = currentUser.name;
            loadOfficerDashboard();
            break;
        case 'client':
            showScreen('clientDashboard');
            document.getElementById('clientName').textContent = currentUser.name;
            loadClientDashboard();
            break;
        case 'accountant':
            showScreen('accountantDashboard');
            document.getElementById('accountantName').textContent = currentUser.name;
            loadAccountantDashboard();
            break;
    }
}

// ============================================
// MANAGER DASHBOARD FUNCTIONS
// ============================================

async function loadManagerDashboard() {
    console.log('Loading manager dashboard...');
    
    // Set default values immediately so something shows up
    document.getElementById('totalClients').textContent = '...';
    document.getElementById('totalLoansDisbursed').textContent = '...';
    document.getElementById('pendingApprovals').textContent = '...';
    document.getElementById('activeCreditOfficers').textContent = '...';
    
    try {
        console.log('Loading manager stats...');
        await loadManagerStats();
        console.log('Manager stats loaded');
        
        console.log('Loading pending loans...');
        await loadPendingLoans();
        console.log('Pending loans loaded');
        
        console.log('Loading manager notifications...');
        await loadManagerNotifications();
        console.log('Manager notifications loaded');
        
        console.log('Manager dashboard fully loaded');
    } catch (error) {
        console.error('Error loading manager dashboard:', error);
        showMessage('Error loading dashboard. Please refresh the page.', 'error');
    }
}

async function loadManagerStats() {
    try {
        console.log('Fetching manager stats from API...');
        const response = await apiCall('/manager/dashboard');
        console.log('Manager stats response:', response);
        
        if (response.success) {
            const data = response.data;
            
            // Update summary cards
            document.getElementById('totalClients').textContent = data.totalClients || 0;
            document.getElementById('totalLoansDisbursed').textContent = data.totalLoansDisbursed || 0;
            document.getElementById('pendingApprovals').textContent = data.pendingApprovals || 0;
            document.getElementById('activeCreditOfficers').textContent = data.activeCreditOfficers || 0;
            
            // Update portfolio
            document.getElementById('portfolioValue').textContent = 
                `UGX ${Number(data.portfolio?.totalValue || 0).toLocaleString()}`;
            document.getElementById('growthPercentage').textContent = `${data.portfolio?.growthPercentage || 0}%`;
            document.getElementById('portfolioGrowth').style.width = 
                `${Math.min(Math.abs(data.portfolio?.growthPercentage || 0), 100)}%`;
            
            // Render monthly performance chart
            if (data.monthlyPerformance && data.monthlyPerformance.length > 0) {
                console.log('Rendering monthly performance chart');
                renderMonthlyPerformanceChart(data.monthlyPerformance);
            } else {
                console.log('No monthly performance data to chart');
            }
        } else {
            console.error('Manager stats response unsuccessful:', response);
        }
    } catch (error) {
        console.error('Error loading manager stats:', error);
        // Don't throw - let other dashboard functions continue
    }
}

// Global chart instance to prevent duplicates
let monthlyPerformanceChart = null;

function renderMonthlyPerformanceChart(data) {
    try {
        const ctx = document.getElementById('loanChart');
        if (!ctx) {
            console.log('Chart canvas not found');
            return;
        }

        // Check if Chart.js is loaded
        if (typeof Chart === 'undefined') {
            console.error('Chart.js library not loaded');
            return;
        }

        // Destroy existing chart if it exists
        if (monthlyPerformanceChart) {
            monthlyPerformanceChart.destroy();
        }

        // Prepare data
        const labels = data.map(item => {
            const [year, month] = item.month.split('-');
            const date = new Date(year, parseInt(month) - 1);
            return date.toLocaleDateString('en-US', { month: 'short', year: 'numeric' });
        });
        const loanCounts = data.map(item => item.loan_count);
        const amounts = data.map(item => item.total_amount / 1000000); // Convert to millions

        monthlyPerformanceChart = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [
                {
                    label: 'Number of Loans',
                    data: loanCounts,
                    backgroundColor: 'rgba(54, 162, 235, 0.5)',
                    borderColor: 'rgba(54, 162, 235, 1)',
                    borderWidth: 2,
                    yAxisID: 'y'
                },
                {
                    label: 'Total Amount (Million UGX)',
                    data: amounts,
                    backgroundColor: 'rgba(75, 192, 192, 0.5)',
                    borderColor: 'rgba(75, 192, 192, 1)',
                    borderWidth: 2,
                    type: 'line',
                    yAxisID: 'y1'
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: true,
            interaction: {
                mode: 'index',
                intersect: false,
            },
            plugins: {
                title: {
                    display: true,
                    text: 'Loan Performance (Last 6 Months)',
                    font: {
                        size: 16
                    }
                },
                legend: {
                    display: true,
                    position: 'top'
                }
            },
            scales: {
                y: {
                    type: 'linear',
                    display: true,
                    position: 'left',
                    title: {
                        display: true,
                        text: 'Number of Loans'
                    },
                    beginAtZero: true
                },
                y1: {
                    type: 'linear',
                    display: true,
                    position: 'right',
                    title: {
                        display: true,
                        text: 'Amount (Million UGX)'
                    },
                    beginAtZero: true,
                    grid: {
                        drawOnChartArea: false
                    }
                }
            }
        }
    });
    } catch (error) {
        console.error('Error rendering chart:', error);
        // Chart failed but don't break the dashboard
    }
}

async function loadPendingLoans() {
    try {
        const response = await apiCall('/manager/pending-loans');
        if (response.success) {
            const tbody = document.getElementById('pendingLoansTable');
            tbody.innerHTML = '';
            
            if (response.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="7" class="text-center">No pending applications</td></tr>';
                return;
            }
            
            response.data.forEach(loan => {
                const lc1LetterLink = loan.lc1_letter_url 
                    ? `<a href="${loan.lc1_letter_url}" target="_blank" class="btn btn-sm btn-secondary">📄 View Letter</a>`
                    : '<span class="badge warning">No Letter</span>';
                
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${loan.client_name}</td>
                    <td>${loan.loan_type}</td>
                    <td>UGX ${Number(loan.amount).toLocaleString()}</td>
                    <td>${loan.officer_name || 'Unassigned'}</td>
                    <td>${lc1LetterLink}</td>
                    <td>${new Date(loan.created_at).toLocaleDateString()}</td>
                    <td>
                        <button class="btn btn-success btn-sm" onclick="approveLoanModal(${loan.id})">Approve</button>
                        <button class="btn btn-danger btn-sm" onclick="rejectLoan(${loan.id})">Reject</button>
                    </td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading pending loans:', error);
    }
}

async function loadManagerNotifications() {
    try {
        const response = await apiCall('/manager/notifications');
        if (response.success) {
            document.getElementById('notifPendingApprovals').textContent = response.data.pendingApprovals;
            document.getElementById('notifOverdueLoans').textContent = response.data.overdueLoans;
        }
    } catch (error) {
        console.error('Error loading notifications:', error);
    }
}

async function clearAllNotifications(role) {
    if (!confirm('Are you sure you want to clear all notifications?\n\nThis action cannot be undone.')) {
        return;
    }
    
    try {
        const endpoint = role === 'client' ? '/client/notifications/clear-all' : 
                        role === 'accountant' ? '/accountant/notifications/clear-all' : 
                        '/manager/notifications/clear-all';
        
        const response = await apiCall(endpoint, 'DELETE');
        if (response.success) {
            alert('All notifications cleared successfully');
            // Reload notifications based on role
            if (role === 'manager') {
                loadManagerNotifications();
            } else if (role === 'accountant') {
                loadAccountantNotifications();
            } else if (role === 'client') {
                loadClientNotifications();
            }
        } else {
            alert(response.message || 'Failed to clear notifications');
        }
    } catch (error) {
        console.error('Error clearing notifications:', error);
        alert('Error clearing notifications: ' + error.message);
    }
}

function showManagerSection(section, event) {
    // Hide all sections
    document.querySelectorAll('#managerDashboard .dashboard-section').forEach(el => {
        el.classList.remove('active');
    });
    
    // Remove active from all links
    document.querySelectorAll('#managerDashboard .sidebar-link').forEach(link => {
        link.classList.remove('active');
    });
    
    // Map section names to element IDs (handle special cases)
    const sectionMap = {
        'summary': 'managerSummary',
        'approvals': 'managerApprovals',
        'enroll': 'managerEnroll',
        'clients': 'managerClients',
        'calendar': 'managerCalendar',
        'financial': 'managerFinancial',
        'pettycash': 'managerPettyCash',
        'settings': 'managerSettings',
        'users': 'managerUsers',
        'profile': 'managerProfile',
        'notifications': 'managerNotifications'
    };
    
    // Get the correct element ID
    const elementId = sectionMap[section] || `manager${section.charAt(0).toUpperCase() + section.slice(1)}`;
    const sectionElement = document.getElementById(elementId);
    
    // Show selected section if it exists
    if (sectionElement) {
        sectionElement.classList.add('active');
    } else {
        console.error(`Section element not found: ${elementId} for section: ${section}`);
        return;
    }
    
    // Set active link if event is provided
    if (event && event.target) {
        event.target.classList.add('active');
    }
    
    // Load data for section
    if (section === 'settings') {
        loadLoanSettings();
        loadSystemSettings();
    } else if (section === 'users') {
        loadUsers();
    } else if (section === 'enroll') {
        loadOfficersDropdown();
    } else if (section === 'clients') {
        loadAllClients();
    } else if (section === 'calendar') {
        loadManagerCalendarOfficers();
        loadManagerPaymentCalendar();
    } else if (section === 'pettycash') {
        loadManagerPettyCash();
    } else if (section === 'financial') {
        loadFinancialOversight();
    } else if (section === 'profile') {
        // Profile section - load user info
        loadManagerProfile();
        setupPasswordChangeForm();
    } else if (section === 'notifications') {
        loadManagerNotifications();
    }
}

function loadManagerProfile() {
    if (currentUser) {
        document.getElementById('managerProfileName').textContent = currentUser.name || 'N/A';
        document.getElementById('managerProfileEmail').textContent = currentUser.email || 'N/A';
        document.getElementById('managerProfileRole').textContent = (currentUser.role || 'manager').toUpperCase();
        document.getElementById('managerProfileId').textContent = currentUser.id || 'N/A';
    }
}

async function loadLoanSettings() {
    try {
        const response = await apiCall('/manager/loan-settings');
        if (response.success) {
            const tbody = document.getElementById('loanSettingsTableBody');
            tbody.innerHTML = '';
            
            if (response.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="5" class="text-center">No loan types configured</td></tr>';
                return;
            }
            
            response.data.forEach(setting => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td><strong>${setting.loan_type}</strong></td>
                    <td>${setting.interest_rate}%</td>
                    <td>${setting.penalty_rate}%</td>
                    <td>${setting.early_payment_bonus}%</td>
                    <td>
                        <button class="btn btn-sm btn-secondary" onclick="editLoanSetting('${setting.loan_type}', ${setting.interest_rate}, ${setting.penalty_rate}, ${setting.early_payment_bonus})">✏️ Edit</button>
                    </td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading loan settings:', error);
        document.getElementById('loanSettingsTableBody').innerHTML = 
            '<tr><td colspan="5" class="text-center error">Failed to load settings</td></tr>';
    }
}

function editLoanSetting(loanType, interestRate, penaltyRate, earlyBonus) {
    const newInterestRate = prompt(`Enter new Interest Rate for ${loanType} (current: ${interestRate}%):`, interestRate);
    const newPenaltyRate = prompt(`Enter new Penalty Rate for ${loanType} (current: ${penaltyRate}%):`, penaltyRate);
    const newEarlyBonus = prompt(`Enter new Early Payment Bonus for ${loanType} (current: ${earlyBonus}%):`, earlyBonus);
    
    if (newInterestRate !== null && newPenaltyRate !== null && newEarlyBonus !== null) {
        updateLoanSettings(loanType, newInterestRate, newPenaltyRate, newEarlyBonus);
    }
}

async function updateLoanSettings(loanType, interestRate, penaltyRate, earlyBonus) {
    try {
        const response = await apiCall('/manager/loan-settings', 'PUT', {
            loan_type: loanType,
            interest_rate: parseFloat(interestRate),
            penalty_rate: parseFloat(penaltyRate),
            early_payment_bonus: parseFloat(earlyBonus)
        });
        
        if (response.success) {
            alert('✅ Settings updated successfully!');
            loadLoanSettings();
        }
    } catch (error) {
        alert('❌ Failed to update settings');
    }
}

// Load system settings (application fee, etc.)
async function loadSystemSettings() {
    try {
        const response = await apiCall('/manager/system-settings');
        if (response.success) {
            const settings = response.data;
            
            // Set application fee amount (processing fee is now automatic based on loan amount)
            if (settings.default_application_fee) {
                document.getElementById('applicationFeeAmount').value = settings.default_application_fee.value;
            }
        }
    } catch (error) {
        console.error('Error loading system settings:', error);
    }
}

// System Configuration Functions
function backupDatabase() {
    const messageDiv = document.getElementById('maintenanceMessage');
    messageDiv.textContent = '📦 Database backup initiated. Please wait...';
    messageDiv.className = 'message info';
    messageDiv.style.display = 'block';
    
    // Simulate backup (in production, this would trigger actual backup)
    setTimeout(() => {
        messageDiv.textContent = '✅ Database backed up successfully! File saved to server.';
        messageDiv.className = 'message success';
    }, 2000);
}

function generateSystemReport() {
    const messageDiv = document.getElementById('maintenanceMessage');
    messageDiv.textContent = '📊 Generating system health report...';
    messageDiv.className = 'message info';
    messageDiv.style.display = 'block';
    
    setTimeout(() => {
        messageDiv.innerHTML = `
            <strong>✅ System Health Report</strong><br>
            • Database: Connected<br>
            • Server Uptime: Active<br>
            • Total Users: ${document.getElementById('totalClients').textContent}<br>
            • Active Loans: ${document.getElementById('totalLoansDisbursed').textContent}<br>
            • System Status: All systems operational
        `;
        messageDiv.className = 'message success';
    }, 1500);
}

function clearCache() {
    if (confirm('Are you sure you want to clear the cache? This will log you out.')) {
        localStorage.clear();
        sessionStorage.clear();
        const messageDiv = document.getElementById('maintenanceMessage');
        messageDiv.textContent = '✅ Cache cleared successfully! Redirecting to login...';
        messageDiv.className = 'message success';
        messageDiv.style.display = 'block';
        
        setTimeout(() => {
            logout();
        }, 2000);
    }
}

async function loadUsers() {
    try {
        const response = await apiCall('/manager/users');
        if (response.success) {
            const tbody = document.getElementById('usersTable');
            tbody.innerHTML = '';
            
            response.data.forEach(user => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${user.name}</td>
                    <td>${user.email}</td>
                    <td>${user.role}</td>
                    <td><span class="status-badge status-${user.status}">${user.status}</span></td>
                    <td>
                        <button class="btn btn-sm" onclick="viewUser(${user.id})" title="View Details">👁️ View</button>
                        <button class="btn btn-sm" onclick="editUser(${user.id})" title="Edit User">✏️ Edit</button>
                        ${user.role === 'client' ? `<button class="btn btn-sm" onclick="reassignUser(${user.id})" title="Reassign Officer">👤 Reassign</button>` : ''}
                        <button class="btn btn-sm" onclick="resetUserPassword(${user.id})" title="Reset Password">🔑 Reset Pwd</button>
                        <button class="btn btn-sm btn-danger" onclick="deleteUser(${user.id}, '${user.name}')" style="margin-left: 5px;" title="Delete User">🗑️ Delete</button>
                    </td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading users:', error);
    }
}

async function deleteUser(userId, userName) {
    if (!confirm(`Are you sure you want to delete user "${userName}"?\n\nThis action cannot be undone. Users with active loans or assigned clients cannot be deleted.`)) {
        return;
    }
    
    try {
        const response = await apiCall(`/manager/users/${userId}`, 'DELETE');
        if (response.success) {
            alert('User deleted successfully');
            loadUsers();
        } else {
            alert(response.message || 'Failed to delete user');
        }
    } catch (error) {
        console.error('Error deleting user:', error);
        alert('Error deleting user: ' + error.message);
    }
}

async function approveLoanModal(loanId) {
    document.getElementById('approveLoanId').value = loanId;
    
    // Fetch loan details to show requested amount
    try {
        const response = await apiCall('/manager/pending-loans');
        if (response.success) {
            const loan = response.data.find(l => l.id === loanId);
            if (loan) {
                document.getElementById('approveRequestedAmount').value = loan.amount;
                document.getElementById('displayRequestedAmount').textContent = `UGX ${Number(loan.amount).toLocaleString()}`;
                document.getElementById('disbursedAmount').value = loan.amount; // Default to requested amount
            }
        }
    } catch (error) {
        console.error('Error loading loan details:', error);
    }
    
    loadOfficersForAssignment();
    openModal('approveLoanModal');
}

async function loadOfficersForAssignment() {
    try {
        const response = await apiCall('/manager/users');
        if (response.success) {
            const select = document.getElementById('assignOfficer');
            select.innerHTML = '<option value="">Select Officer</option>';
            
            response.data.filter(u => u.role === 'officer').forEach(officer => {
                const option = document.createElement('option');
                option.value = officer.id;
                option.textContent = officer.name;
                select.appendChild(option);
            });
        }
    } catch (error) {
        console.error('Error loading officers:', error);
    }
}

async function rejectLoan(loanId) {
    if (!confirm('Are you sure you want to reject this loan?')) return;
    
    try {
        const response = await apiCall(`/manager/loans/${loanId}/status`, 'PUT', {
            status: 'rejected'
        });
        
        if (response.success) {
            alert('Loan rejected');
            loadPendingLoans();
        }
    } catch (error) {
        alert('Failed to reject loan');
    }
}

function showCreateUserModal() {
    openModal('createUserModal');
}

async function viewUser(userId) {
    try {
        const response = await apiCall(`/manager/users/${userId}`);
        if (response.success) {
            const user = response.data;
            alert(`User Details:\n\nName: ${user.name}\nEmail: ${user.email}\nPhone: ${user.phone}\nRole: ${user.role}\nStatus: ${user.status}\nCreated: ${new Date(user.created_at).toLocaleString()}`);
        }
    } catch (error) {
        alert('Failed to load user details');
    }
}

async function editUser(userId) {
    try {
        const response = await apiCall(`/manager/users/${userId}`);
        if (response.success) {
            const user = response.data;
            const name = prompt('Enter new name:', user.name);
            if (!name) return;
            
            const email = prompt('Enter new email:', user.email);
            if (!email) return;
            
            const phone = prompt('Enter new phone:', user.phone);
            if (!phone) return;
            
            const role = prompt('Enter new role (manager/accountant/officer/client):', user.role);
            if (!role) return;
            
            const status = prompt('Enter status (active/inactive):', user.status);
            if (!status) return;
            
            const updateResponse = await apiCall(`/manager/users/${userId}`, 'PUT', {
                name,
                email,
                phone,
                role,
                status
            });
            
            if (updateResponse.success) {
                alert('✅ User updated successfully!');
                loadUsers();
            } else {
                alert('❌ Failed to update user: ' + (updateResponse.message || 'Unknown error'));
            }
        }
    } catch (error) {
        alert('❌ Failed to edit user: ' + error.message);
    }
}

async function reassignUser(userId) {
    try {
        // Get list of officers
        const officersResponse = await apiCall('/manager/officers');
        if (!officersResponse.success) {
            alert('Failed to load officers');
            return;
        }
        
        const officers = officersResponse.officers || [];
        if (officers.length === 0) {
            alert('No officers available');
            return;
        }
        
        let officerList = 'Available Officers:\n\n';
        officers.forEach(officer => {
            officerList += `ID: ${officer.id} - ${officer.name} (${officer.email})\n`;
        });
        
        const officerId = prompt(officerList + '\nEnter Officer ID to assign:');
        if (!officerId) return;
        
        const response = await apiCall(`/manager/clients/${userId}/assign`, 'PUT', {
            officer_id: parseInt(officerId)
        });
        
        if (response.success) {
            alert('✅ Client reassigned successfully!');
            loadUsers();
        } else {
            alert('❌ Failed to reassign client: ' + (response.message || 'Unknown error'));
        }
    } catch (error) {
        alert('❌ Failed to reassign user: ' + error.message);
    }
}

async function resetUserPassword(userId) {
    const newPassword = prompt('Enter new password:');
    if (!newPassword) return;
    
    try {
        const response = await apiCall(`/manager/users/${userId}/reset-password`, 'PUT', { newPassword });
        if (response.success) {
            alert('Password reset successfully');
        }
    } catch (error) {
        alert('Failed to reset password');
    }
}

// ============================================
// CREDIT OFFICER DASHBOARD FUNCTIONS
// ============================================

async function loadOfficerDashboard() {
    await loadAssignedClients();
    await loadDailyTasks();
    await loadOfficerNotifications();
}

async function loadOfficerNotifications() {
    try {
        const response = await apiCall('/officer/notifications');
        if (response.success) {
            const container = document.getElementById('officerNotificationsList');
            if (!container) return;
            
            container.innerHTML = '';
            
            if (response.data.length === 0) {
                container.innerHTML = '<p class="text-muted">No notifications</p>';
                return;
            }
            
            response.data.forEach(notif => {
                const notifDiv = document.createElement('div');
                notifDiv.className = `notification-item ${notif.is_read ? 'read' : 'unread'}`;
                notifDiv.innerHTML = `
                    <div class="notification-content">
                        <span class="notification-type">${notif.type}</span>
                        <p>${notif.message}</p>
                        <small>${new Date(notif.created_at).toLocaleString()}</small>
                    </div>
                    ${!notif.is_read ? `<button class="btn btn-sm btn-secondary" onclick="markOfficerNotificationRead(${notif.id})">Mark as Read</button>` : ''}
                `;
                container.appendChild(notifDiv);
            });
        }
    } catch (error) {
        console.error('Error loading officer notifications:', error);
    }
}

async function markOfficerNotificationRead(notificationId) {
    try {
        const response = await apiCall(`/officer/notifications/${notificationId}/read`, 'PUT');
        if (response.success) {
            await loadOfficerNotifications();
        }
    } catch (error) {
        console.error('Error marking notification as read:', error);
    }
}

async function loadAssignedClients() {
    try {
        const response = await apiCall('/officer/clients');
        if (response.success) {
            const tbody = document.getElementById('assignedClientsTable');
            tbody.innerHTML = '';
            
            if (response.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="7" class="text-center">No clients assigned</td></tr>';
                return;
            }
            
            response.data.forEach(client => {
                const row = document.createElement('tr');
                
                // Determine status styling
                let statusClass = 'status-badge';
                let statusStyle = '';
                
                if (client.status === 'pending') {
                    statusClass += ' status-pending';
                    statusStyle = 'background: #fff3cd; color: #856404; font-weight: bold; border: 2px solid #ffc107;';
                    // Highlight new applications
                    row.style.backgroundColor = '#fffbea';
                } else if (client.status === 'approved') {
                    statusClass += ' status-approved';
                    statusStyle = 'background: #d4edda; color: #155724; font-weight: bold;';
                } else if (client.payment_status === 'Overdue') {
                    statusClass += ' status-overdue';
                } else if (client.payment_status === 'Due Today') {
                    statusClass += ' status-warning';
                } else if (client.status === 'rejected') {
                    statusClass += ' status-rejected';
                } else {
                    statusClass += ' status-active';
                }
                
                // Format application date
                const appDate = client.created_at ? new Date(client.created_at).toLocaleDateString() : 'N/A';
                
                // Show different info based on loan status
                let actionButton = '';
                if (client.status === 'pending') {
                    actionButton = `<button class="btn btn-sm btn-primary" onclick="viewLoanApplication(${client.loan_id})">📋 Review Application</button>`;
                } else if (client.loan_id) {
                    actionButton = `<button class="btn btn-sm" onclick="viewClientLoanDetails(${client.client_id}, ${client.loan_id})">View Loan</button>`;
                } else {
                    actionButton = `<button class="btn btn-sm" onclick="viewClientDetails(${client.client_id})">View Client</button>`;
                }
                
                row.innerHTML = `
                    <td><strong>${client.client_name}</strong></td>
                    <td>${client.phone || 'N/A'}</td>
                    <td>${client.loan_type || 'No Loan'}</td>
                    <td><strong>UGX ${client.amount ? Number(client.amount).toLocaleString() : '0'}</strong></td>
                    <td><span class="${statusClass}" style="${statusStyle}">${client.payment_status}</span></td>
                    <td>${appDate}</td>
                    <td>${actionButton}</td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading assigned clients:', error);
    }
}

async function loadDailyTasks() {
    try {
        const response = await apiCall('/officer/daily-tasks');
        if (response.success) {
            document.getElementById('overdueClientsCount').textContent = response.data.overdueCount;
            
            const tbody = document.getElementById('dailyTasksTable');
            tbody.innerHTML = '';
            
            if (response.data.tasks.length === 0) {
                tbody.innerHTML = '<tr><td colspan="7" class="text-center">No tasks for today</td></tr>';
                return;
            }
            
            response.data.tasks.forEach(task => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${task.client_name}</td>
                    <td>${task.phone}</td>
                    <td>${task.loan_type}</td>
                    <td>UGX ${Number(task.balance).toLocaleString()}</td>
                    <td>${task.next_payment_date}</td>
                    <td>${task.days_overdue > 0 ? task.days_overdue : 'Due Today'}</td>
                    <td>
                        <button class="btn btn-sm" onclick="callClient('${task.phone}')">Call</button>
                    </td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading daily tasks:', error);
    }
}

function showOfficerSection(section, event) {
    document.querySelectorAll('#officerDashboard .dashboard-section').forEach(el => {
        el.classList.remove('active');
    });
    
    document.querySelectorAll('#officerDashboard .sidebar-link').forEach(link => {
        link.classList.remove('active');
    });
    
    document.getElementById(`officer${section.charAt(0).toUpperCase() + section.slice(1)}`).classList.add('active');
    if (event && event.target) {
        event.target.classList.add('active');
    }
    
    if (section === 'tasks') {
        loadDailyTasks();
    } else if (section === 'calendar') {
        loadPaymentCalendar();
    }
}

// Call client function
function callClient(phone) {
    if (!phone) {
        alert('Phone number not available');
        return;
    }
    
    // Format phone number (remove spaces and non-numeric characters)
    const cleanPhone = phone.replace(/\D/g, '');
    
    // Confirm before calling
    if (confirm(`Call ${phone}?\n\nThis will open your phone app or dialer.`)) {
        // Use tel: protocol to open phone dialer
        window.location.href = `tel:${cleanPhone}`;
    }
}

// Generate officer report
async function generateOfficerReport(period) {
    try {
        const response = await apiCall(`/officer/generate-report?period=${period}`);
        
        if (response.success) {
            const data = response.data;
            const reportTitle = period === 'month' ? 'Monthly Report' : 'Weekly Report';
            const reportDate = new Date().toLocaleDateString('en-GB', { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric' 
            });
            const officerName = localStorage.getItem('userName') || 'Officer';
            
            // Create a new window for the report
            const printWindow = window.open('', '_blank');
            printWindow.document.write(`
                <!DOCTYPE html>
                <html>
                <head>
                    <title>${reportTitle} - Starting Point Financial Institution</title>
                    <style>
                        * {
                            margin: 0;
                            padding: 0;
                            box-sizing: border-box;
                        }
                        body {
                            font-family: 'Arial', sans-serif;
                            padding: 40px;
                            background: white;
                        }
                        .header {
                            text-align: center;
                            margin-bottom: 30px;
                            border-bottom: 3px solid #2563eb;
                            padding-bottom: 20px;
                        }
                        .header h1 {
                            color: #2563eb;
                            font-size: 28px;
                            margin-bottom: 10px;
                        }
                        .header h2 {
                            color: #333;
                            font-size: 20px;
                            font-weight: normal;
                        }
                        .info-section {
                            display: flex;
                            justify-content: space-between;
                            margin: 20px 0;
                            padding: 15px;
                            background: #f9fafb;
                            border-radius: 8px;
                        }
                        .info-item {
                            flex: 1;
                        }
                        .info-label {
                            font-weight: bold;
                            color: #666;
                            font-size: 12px;
                            text-transform: uppercase;
                            margin-bottom: 5px;
                        }
                        .info-value {
                            color: #333;
                            font-size: 14px;
                        }
                        .stats-grid {
                            display: grid;
                            grid-template-columns: repeat(2, 1fr);
                            gap: 20px;
                            margin: 30px 0;
                        }
                        .stat-card {
                            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                            color: white;
                            padding: 30px;
                            border-radius: 12px;
                            text-align: center;
                            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
                        }
                        .stat-card:nth-child(2) {
                            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
                        }
                        .stat-card:nth-child(3) {
                            background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
                        }
                        .stat-card:nth-child(4) {
                            background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);
                        }
                        .stat-label {
                            font-size: 14px;
                            opacity: 0.9;
                            margin-bottom: 10px;
                            text-transform: uppercase;
                            letter-spacing: 1px;
                        }
                        .stat-value {
                            font-size: 36px;
                            font-weight: bold;
                        }
                        .footer {
                            margin-top: 40px;
                            padding-top: 20px;
                            border-top: 2px solid #e5e7eb;
                            text-align: center;
                            color: #666;
                            font-size: 12px;
                        }
                        .action-buttons {
                            text-align: center;
                            margin: 30px 0;
                        }
                        .btn {
                            padding: 12px 30px;
                            margin: 0 10px;
                            font-size: 16px;
                            border: none;
                            border-radius: 6px;
                            cursor: pointer;
                            font-weight: bold;
                        }
                        .btn-primary {
                            background: #2563eb;
                            color: white;
                        }
                        .btn-secondary {
                            background: #6b7280;
                            color: white;
                        }
                        @media print {
                            .action-buttons {
                                display: none;
                            }
                            body {
                                padding: 20px;
                            }
                        }
                    </style>
                </head>
                <body>
                    <div class="header">
                        <h1>Starting Point Financial Institution</h1>
                        <h2>${reportTitle}</h2>
                    </div>
                    
                    <div class="info-section">
                        <div class="info-item">
                            <div class="info-label">Officer</div>
                            <div class="info-value">${officerName}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">Report Period</div>
                            <div class="info-value">${period === 'month' ? 'Last 30 Days' : 'Last 7 Days'}</div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">Generated On</div>
                            <div class="info-value">${reportDate}</div>
                        </div>
                    </div>
                    
                    <div class="stats-grid">
                        <div class="stat-card">
                            <div class="stat-label">Total Clients</div>
                            <div class="stat-value">${data.total_clients || 0}</div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-label">Total Payments</div>
                            <div class="stat-value">${data.total_payments || 0}</div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-label">Total Collected</div>
                            <div class="stat-value">UGX ${Number(data.total_collected || 0).toLocaleString()}</div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-label">Overdue Loans</div>
                            <div class="stat-value">${data.overdue_loans || 0}</div>
                        </div>
                    </div>
                    
                    <div class="action-buttons">
                        <button onclick="window.print()" class="btn btn-primary">🖨️ Print / Save as PDF</button>
                        <button onclick="window.close()" class="btn btn-secondary">Close</button>
                    </div>
                    
                    <div class="footer">
                        <p><strong>Starting Point Financial Institution</strong></p>
                        <p>This is a computer-generated report. No signature is required.</p>
                        <p>Generated on ${reportDate} at ${new Date().toLocaleTimeString()}</p>
                    </div>
                </body>
                </html>
            `);
            printWindow.document.close();
            
            // Auto-focus the print window
            printWindow.focus();
        } else {
            alert('Failed to generate report: ' + (response.message || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error generating report:', error);
        alert('Error generating report. Please try again.');
    }
}

// Payment Calendar Functions
let currentCalendarDate = new Date();
let calendarPayments = [];

async function loadPaymentCalendar() {
    try {
        const response = await apiCall('/officer/clients');
        if (response.success) {
            // Extract all payments with dates
            calendarPayments = response.data
                .filter(client => client.next_payment_date && client.balance > 0)
                .map(client => ({
                    date: client.next_payment_date,
                    clientName: client.client_name,
                    phone: client.phone,
                    amount: client.balance,
                    loanType: client.loan_type,
                    status: client.payment_status
                }));
            
            renderCalendar();
        }
    } catch (error) {
        console.error('Error loading payment calendar:', error);
    }
}

function renderCalendar() {
    const year = currentCalendarDate.getFullYear();
    const month = currentCalendarDate.getMonth();
    
    // Update month/year display
    const monthNames = ['January', 'February', 'March', 'April', 'May', 'June',
                        'July', 'August', 'September', 'October', 'November', 'December'];
    document.getElementById('calendarMonthYear').textContent = `${monthNames[month]} ${year}`;
    
    // Get first day of month and number of days
    const firstDay = new Date(year, month, 1).getDay();
    const daysInMonth = new Date(year, month + 1, 0).getDate();
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    
    const grid = document.getElementById('paymentCalendarGrid');
    grid.innerHTML = '';
    
    // Day headers
    const dayNames = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
    dayNames.forEach(day => {
        const header = document.createElement('div');
        header.textContent = day;
        header.style.fontWeight = 'bold';
        header.style.textAlign = 'center';
        header.style.padding = '10px';
        header.style.background = '#f0f0f0';
        header.style.borderRadius = '4px';
        grid.appendChild(header);
    });
    
    // Empty cells before first day
    for (let i = 0; i < firstDay; i++) {
        const empty = document.createElement('div');
        empty.style.minHeight = '80px';
        grid.appendChild(empty);
    }
    
    // Days of month
    for (let day = 1; day <= daysInMonth; day++) {
        const dateStr = `${year}-${String(month + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
        const cellDate = new Date(year, month, day);
        cellDate.setHours(0, 0, 0, 0);
        
        // Find payments for this date
        const dayPayments = calendarPayments.filter(p => p.date === dateStr);
        
        // Determine cell style
        let bgColor = '#fff';
        let borderColor = '#ddd';
        let textColor = '#333';
        
        if (dayPayments.length > 0) {
            const isOverdue = cellDate < today;
            const isToday = cellDate.getTime() === today.getTime();
            
            if (isOverdue) {
                bgColor = '#ffe6e6';
                borderColor = '#dc3545';
                textColor = '#dc3545';
            } else if (isToday) {
                bgColor = '#fff3cd';
                borderColor = '#ffc107';
                textColor = '#856404';
            } else {
                bgColor = '#d4edda';
                borderColor = '#28a745';
                textColor = '#155724';
            }
        }
        
        const cell = document.createElement('div');
        cell.style.minHeight = '80px';
        cell.style.padding = '8px';
        cell.style.background = bgColor;
        cell.style.border = `2px solid ${borderColor}`;
        cell.style.borderRadius = '8px';
        cell.style.cursor = dayPayments.length > 0 ? 'pointer' : 'default';
        cell.style.position = 'relative';
        
        cell.innerHTML = `
            <div style="font-weight: bold; color: ${textColor};">${day}</div>
            ${dayPayments.length > 0 ? `<div style="font-size: 0.85em; color: ${textColor}; margin-top: 4px;">${dayPayments.length} payment${dayPayments.length > 1 ? 's' : ''}</div>` : ''}
        `;
        
        if (dayPayments.length > 0) {
            cell.onclick = () => showDatePayments(dateStr, dayPayments);
        }
        
        grid.appendChild(cell);
    }
}

function changeCalendarMonth(direction) {
    currentCalendarDate.setMonth(currentCalendarDate.getMonth() + direction);
    renderCalendar();
}

function showDatePayments(dateStr, payments) {
    const formatted = new Date(dateStr).toLocaleDateString('en-GB', { 
        weekday: 'long', 
        year: 'numeric', 
        month: 'long', 
        day: 'numeric' 
    });
    
    document.getElementById('selectedDateTitle').textContent = formatted;
    document.getElementById('selectedDatePaymentsList').innerHTML = payments.map(p => `
        <div style="padding: 10px; background: white; border-radius: 8px; margin-bottom: 10px; border-left: 4px solid #2563eb;">
            <div style="font-weight: bold;">${p.clientName}</div>
            <div style="font-size: 0.9em; color: #666;">
                ${p.loanType} - UGX ${Number(p.amount).toLocaleString()}
            </div>
            <div style="margin-top: 5px;">
                <a href="tel:${p.phone}" class="btn btn-sm" style="display: inline-block; padding: 4px 8px; font-size: 0.85em;">📞 Call ${p.phone}</a>
            </div>
        </div>
    `).join('');
    
    document.getElementById('selectedDatePayments').style.display = 'block';
    document.getElementById('selectedDatePayments').scrollIntoView({ behavior: 'smooth' });
}

// Manager Payment Calendar Functions
let currentManagerCalendarDate = new Date();
let managerCalendarPayments = [];

async function loadManagerCalendarOfficers() {
    try {
        const response = await apiCall('/manager/officers');
        if (response.success) {
            const select = document.getElementById('managerCalendarOfficerFilter');
            select.innerHTML = '<option value="">All Officers</option>';
            response.officers.forEach(officer => {
                const option = document.createElement('option');
                option.value = officer.id;
                option.textContent = officer.name;
                select.appendChild(option);
            });
        }
    } catch (error) {
        console.error('Error loading officers for calendar:', error);
    }
}

async function loadManagerPaymentCalendar() {
    try {
        const officerId = document.getElementById('managerCalendarOfficerFilter').value;
        const response = await apiCall('/enrollment/clients');
        
        if (response.success) {
            // Extract all payments with dates
            let clients = response.clients;
            
            // Filter by officer if selected
            if (officerId) {
                clients = clients.filter(c => c.officer_id == officerId);
            }
            
            // Get all clients with loans
            const allPayments = [];
            for (const client of clients) {
                try {
                    const clientResponse = await apiCall(`/enrollment/clients/${client.id}`);
                    if (clientResponse.success && clientResponse.client) {
                        const clientData = clientResponse.client;
                        // Get loans for this client
                        const loansResponse = await apiCall(`/enrollment/clients/${client.id}/loans`);
                        if (loansResponse.success && loansResponse.loans) {
                            loansResponse.loans.forEach(loan => {
                                if (loan.next_payment_date && loan.balance > 0 && loan.status === 'disbursed') {
                                    // Convert timestamp to YYYY-MM-DD format
                                    const paymentDate = new Date(loan.next_payment_date);
                                    const dateStr = `${paymentDate.getFullYear()}-${String(paymentDate.getMonth() + 1).padStart(2, '0')}-${String(paymentDate.getDate()).padStart(2, '0')}`;
                                    
                                    allPayments.push({
                                        date: dateStr,
                                        clientName: clientData.name,
                                        phone: clientData.phone,
                                        amount: loan.balance,
                                        loanType: loan.loan_type,
                                        officerName: clientData.officer_name || 'Unassigned',
                                        status: loan.payment_status
                                    });
                                }
                            });
                        }
                    }
                } catch (err) {
                    console.log('Could not load loans for client', client.id);
                }
            }
            
            managerCalendarPayments = allPayments;
            renderManagerCalendar();
        }
    } catch (error) {
        console.error('Error loading manager payment calendar:', error);
    }
}

function renderManagerCalendar() {
    const year = currentManagerCalendarDate.getFullYear();
    const month = currentManagerCalendarDate.getMonth();
    
    // Update month/year display
    const monthNames = ['January', 'February', 'March', 'April', 'May', 'June',
                        'July', 'August', 'September', 'October', 'November', 'December'];
    document.getElementById('managerCalendarMonthYear').textContent = `${monthNames[month]} ${year}`;
    
    // Get first day of month and number of days
    const firstDay = new Date(year, month, 1).getDay();
    const daysInMonth = new Date(year, month + 1, 0).getDate();
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    
    const grid = document.getElementById('managerPaymentCalendarGrid');
    grid.innerHTML = '';
    
    // Day headers
    const dayNames = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
    dayNames.forEach(day => {
        const header = document.createElement('div');
        header.textContent = day;
        header.style.fontWeight = 'bold';
        header.style.textAlign = 'center';
        header.style.padding = '10px';
        header.style.background = '#f0f0f0';
        header.style.borderRadius = '4px';
        grid.appendChild(header);
    });
    
    // Empty cells before first day
    for (let i = 0; i < firstDay; i++) {
        const empty = document.createElement('div');
        empty.style.minHeight = '80px';
        grid.appendChild(empty);
    }
    
    // Days of month
    for (let day = 1; day <= daysInMonth; day++) {
        const dateStr = `${year}-${String(month + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
        const cellDate = new Date(year, month, day);
        cellDate.setHours(0, 0, 0, 0);
        
        // Find payments for this date
        const dayPayments = managerCalendarPayments.filter(p => p.date === dateStr);
        
        // Determine cell style
        let bgColor = '#fff';
        let borderColor = '#ddd';
        let textColor = '#333';
        
        if (dayPayments.length > 0) {
            const isOverdue = cellDate < today;
            const isToday = cellDate.getTime() === today.getTime();
            
            if (isOverdue) {
                bgColor = '#ffe6e6';
                borderColor = '#dc3545';
                textColor = '#dc3545';
            } else if (isToday) {
                bgColor = '#fff3cd';
                borderColor = '#ffc107';
                textColor = '#856404';
            } else {
                bgColor = '#d4edda';
                borderColor = '#28a745';
                textColor = '#155724';
            }
        }
        
        const cell = document.createElement('div');
        cell.style.minHeight = '80px';
        cell.style.padding = '8px';
        cell.style.background = bgColor;
        cell.style.border = `2px solid ${borderColor}`;
        cell.style.borderRadius = '8px';
        cell.style.cursor = dayPayments.length > 0 ? 'pointer' : 'default';
        cell.style.position = 'relative';
        
        cell.innerHTML = `
            <div style="font-weight: bold; color: ${textColor};">${day}</div>
            ${dayPayments.length > 0 ? `<div style="font-size: 0.85em; color: ${textColor}; margin-top: 4px;">${dayPayments.length} payment${dayPayments.length > 1 ? 's' : ''}</div>` : ''}
        `;
        
        if (dayPayments.length > 0) {
            cell.onclick = () => showManagerDatePayments(dateStr, dayPayments);
        }
        
        grid.appendChild(cell);
    }
}

function changeManagerCalendarMonth(direction) {
    currentManagerCalendarDate.setMonth(currentManagerCalendarDate.getMonth() + direction);
    renderManagerCalendar();
}

function showManagerDatePayments(dateStr, payments) {
    const formatted = new Date(dateStr).toLocaleDateString('en-GB', { 
        weekday: 'long', 
        year: 'numeric', 
        month: 'long', 
        day: 'numeric' 
    });
    
    document.getElementById('managerSelectedDateTitle').textContent = formatted;
    document.getElementById('managerSelectedDatePaymentsList').innerHTML = payments.map(p => `
        <div style="padding: 10px; background: white; border-radius: 8px; margin-bottom: 10px; border-left: 4px solid #2563eb;">
            <div style="font-weight: bold;">${p.clientName}</div>
            <div style="font-size: 0.9em; color: #666;">
                ${p.loanType} - UGX ${Number(p.amount).toLocaleString()}
            </div>
            <div style="font-size: 0.85em; color: #666; margin-top: 5px;">
                👤 Officer: ${p.officerName}
            </div>
            <div style="margin-top: 5px;">
                <a href="tel:${p.phone}" class="btn btn-sm" style="display: inline-block; padding: 4px 8px; font-size: 0.85em;">📞 Call ${p.phone}</a>
            </div>
        </div>
    `).join('');
    
    document.getElementById('managerSelectedDatePayments').style.display = 'block';
    document.getElementById('managerSelectedDatePayments').scrollIntoView({ behavior: 'smooth' });
}

async function loadManagerPettyCash() {
    try {
        const response = await apiCall('/manager/petty-cash');
        if (response.success) {
            const tbody = document.getElementById('managerPettyCashTable');
            tbody.innerHTML = '';
            
            if (response.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="4" class="text-center">No petty cash records</td></tr>';
                return;
            }
            
            response.data.forEach(cash => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${new Date(cash.transaction_date).toLocaleDateString()}</td>
                    <td>UGX ${Number(cash.amount).toLocaleString()}</td>
                    <td>${cash.description}</td>
                    <td>${cash.recorded_by_name || 'N/A'}</td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading manager petty cash:', error);
    }
}

// Load Financial Oversight (Manager)
async function loadFinancialOversight() {
    try {
        const response = await apiCall('/manager/financial-oversight');
        if (response.success) {
            const { accountBalances, pettyCash, loanStats, allocationHistory, summary } = response.data;
            
            // Populate account balances (Cash, Bank, Mobile Money, Petty Cash, Loans)
            if (accountBalances) {
                accountBalances.forEach(account => {
                    const balance = Number(account.balance).toLocaleString();
                    switch(account.account_code) {
                        case '1110':
                            const cashEl = document.getElementById('managerCashOnHand');
                            if (cashEl) cashEl.textContent = `UGX ${balance}`;
                            break;
                        case '1120':
                            const bankEl = document.getElementById('managerBankAccount');
                            if (bankEl) bankEl.textContent = `UGX ${balance}`;
                            break;
                        case '1130':
                            const mobileEl = document.getElementById('managerMobileMoney');
                            if (mobileEl) mobileEl.textContent = `UGX ${balance}`;
                            break;
                        case '1140':
                            const pettyEl = document.getElementById('managerPettyCashCard');
                            if (pettyEl) pettyEl.textContent = `UGX ${balance}`;
                            break;
                        case '1210':
                            const loanEl = document.getElementById('managerLoanReceivables');
                            if (loanEl) loanEl.textContent = `UGX ${balance}`;
                            break;
                    }
                });
            }
            
            // Populate loan statistics
            if (loanStats) {
                const disbEl = document.getElementById('managerTotalDisbursed');
                const balEl = document.getElementById('managerOutstandingBalance');
                
                if (disbEl) disbEl.textContent = `UGX ${Number(loanStats.totalDisbursed).toLocaleString()}`;
                if (balEl) balEl.textContent = `UGX ${Number(loanStats.outstandingBalance).toLocaleString()}`;
            }
            
            // Hide account allocations section (not available in current database structure)
            const accountContainer = document.getElementById('managerAccountAllocations');
            if (accountContainer) {
                accountContainer.parentElement.style.display = 'none';
            }
            
            // Populate petty cash overview
            const pettyCashContainer = document.getElementById('managerPettyCashOverview');
            if (pettyCashContainer && pettyCash) {
                const pettyCashPercent = pettyCash.allocated_amount > 0 
                    ? ((pettyCash.spent / pettyCash.allocated_amount) * 100).toFixed(1) 
                    : 0;
                
                pettyCashContainer.innerHTML = `
                    <div style="border: 1px solid #ddd; border-radius: 8px; padding: 16px; text-align: center; background: #eff6ff;">
                        <div style="font-size: 0.875rem; color: #666; margin-bottom: 8px;">Allocated</div>
                        <div style="font-size: 1.5rem; font-weight: bold; color: #2563eb;">UGX ${Number(pettyCash.allocated_amount).toLocaleString()}</div>
                    </div>
                    <div style="border: 1px solid #ddd; border-radius: 8px; padding: 16px; text-align: center; background: #f0fdf4;">
                        <div style="font-size: 0.875rem; color: #666; margin-bottom: 8px;">Available</div>
                        <div style="font-size: 1.5rem; font-weight: bold; color: #059669;">UGX ${Number(pettyCash.current_balance).toLocaleString()}</div>
                    </div>
                    <div style="border: 1px solid #ddd; border-radius: 8px; padding: 16px; text-align: center; background: #fef2f2;">
                        <div style="font-size: 0.875rem; color: #666; margin-bottom: 8px;">Used (${pettyCashPercent}%)</div>
                        <div style="font-size: 1.5rem; font-weight: bold; color: #dc2626;">UGX ${Number(pettyCash.spent).toLocaleString()}</div>
                    </div>
                `;
            }
            
            // Populate allocation history
            const historyTable = document.getElementById('allocationHistoryTable');
            if (historyTable && allocationHistory) {
                historyTable.innerHTML = allocationHistory.length === 0 
                    ? '<tr><td colspan="6" class="text-center">No allocation history</td></tr>'
                    : allocationHistory.map(item => `
                        <tr>
                            <td>${new Date(item.date).toLocaleDateString()}</td>
                            <td>${item.account}</td>
                            <td><span style="padding: 4px 8px; border-radius: 4px; font-size: 0.875rem; background: ${item.action === 'allocation' ? '#dbeafe' : '#fef3c7'}; color: ${item.action === 'allocation' ? '#1e40af' : '#92400e'};">${item.action}</span></td>
                            <td>UGX ${Number(item.amount).toLocaleString()}</td>
                            <td>${item.description || 'N/A'}</td>
                            <td>${item.recorded_by || 'System'}</td>
                        </tr>
                    `).join('');
            }
            
            // Update summary cards
            if (summary) {
                const totalAllocEl = document.getElementById('totalAllocated');
                const totalAvailEl = document.getElementById('totalAvailable');
                const totalUsedEl = document.getElementById('totalUsed');
                const utilRateEl = document.getElementById('utilizationRate');
                
                if (totalAllocEl) totalAllocEl.textContent = `UGX ${Number(summary.totalAllocated).toLocaleString()}`;
                if (totalAvailEl) totalAvailEl.textContent = `UGX ${Number(summary.totalAvailable).toLocaleString()}`;
                if (totalUsedEl) totalUsedEl.textContent = `UGX ${Number(summary.totalUsed).toLocaleString()}`;
                if (utilRateEl) utilRateEl.textContent = `${summary.utilizationRate}%`;
            }
        }
    } catch (error) {
        console.error('Error loading financial oversight:', error);
        alert('Failed to load financial oversight data. Please try again.');
    }
}

// Search client for payment recording
async function searchClientForPayment() {
    const searchTerm = document.getElementById('searchClientPayment').value.trim();
    
    if (!searchTerm) {
        alert('⚠️ Please enter a search term');
        return;
    }
    
    try {
        const response = await apiCall(`/accountant/search-loans?search=${encodeURIComponent(searchTerm)}`);
        
        if (response.success) {
            const resultsDiv = document.getElementById('clientLoanResults');
            const tbody = document.getElementById('clientLoansTable');
            tbody.innerHTML = '';
            
            if (response.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="6" class="text-center">No active loans found for this client</td></tr>';
                resultsDiv.style.display = 'block';
                return;
            }
            
            response.data.forEach(loan => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${loan.client_name}<br><small>${loan.client_phone}</small></td>
                    <td>${loan.loan_type}</td>
                    <td>UGX ${Number(loan.requested_amount || loan.amount).toLocaleString()}</td>
                    <td>UGX ${Number(loan.balance).toLocaleString()}</td>
                    <td>${loan.next_payment_date ? new Date(loan.next_payment_date).toLocaleDateString() : 'N/A'}</td>
                    <td>
                        <button class="btn btn-sm btn-primary" onclick="showAccountantRecordPaymentModal(${loan.id}, '${loan.client_name}', ${loan.balance})">💳 Payment</button>
                        <button class="btn btn-sm btn-secondary" onclick="showPaymentHistory(${loan.id})">📋 History</button>
                    </td>
                `;
                tbody.appendChild(row);
            });
            
            resultsDiv.style.display = 'block';
        }
    } catch (error) {
        alert('❌ Failed to search for loans');
        console.error('Error searching loans:', error);
    }
}

function showAllocatePettyCashModal() {
    openModal('allocatePettyCashModal');
}

function showAllocateAccountModal(accountCode, accountName) {
    document.getElementById('accountCodeInput').value = accountCode;
    document.getElementById('accountNameDisplay').value = accountName;
    openModal('allocateAccountModal');
}

// ============================================
// CLIENT DASHBOARD FUNCTIONS
// ============================================

async function loadClientDashboard() {
    await loadLoanOverview();
    await loadRepaymentProgress();
    await loadPortfolioGrowth();
    await loadClientNotifications();
}

async function loadLoanOverview() {
    try {
        const response = await apiCall('/client/loan-overview');
        const content = document.getElementById('loanOverviewContent');
        const topupButton = document.querySelector('#clientOverview .btn-primary');
        
        if (response.success) {
            // Check if new client
            if (response.isNewClient || !response.data) {
                content.innerHTML = `
                    <div class="info-message" style="background: #fff3cd; padding: 20px; border-radius: 8px; border-left: 4px solid #ffc107;">
                        <h3>👋 Welcome to SPFI Loan Management System</h3>
                        <p>You don't have any loan applications yet. Please visit your assigned credit officer to apply for your first loan.</p>
                        <p style="margin-top: 15px;"><strong>What's Next?</strong></p>
                        <ul style="margin-left: 20px;">
                            <li>Contact your credit officer to discuss loan options</li>
                            <li>Prepare required documents (ID, collateral information)</li>
                            <li>Submit your loan application through your officer</li>
                        </ul>
                    </div>
                `;
                // Hide top-up button for new clients
                if (topupButton) topupButton.style.display = 'none';
                window.currentClientLoan = null;
                return;
            }
            
            const loan = response.data;
            window.currentClientLoan = loan;
            
            // Display loan information
            content.innerHTML = `
                <div class="detail-item">
                    <span class="label">Loan Type:</span>
                    <span class="value">${loan.loan_type}</span>
                </div>
                <div class="detail-item">
                    <span class="label">Requested Amount:</span>
                    <span class="value">UGX ${Number(loan.amount).toLocaleString()}</span>
                </div>
                <div class="detail-item">
                    <span class="label">Disbursed Amount:
                        <span title="This is the amount you actually receive after the processing fee is deducted from your loan request." style="cursor: help; color: #17a2b8;">&#9432;</span>
                    </span>
                    <span class="value">UGX ${Number(loan.disbursed_amount || 0).toLocaleString()}</span>
                </div>
                <div class="detail-item">
                    <span class="label">Amount Paid:</span>
                    <span class="value">UGX ${Number(loan.amount_paid || 0).toLocaleString()}</span>
                </div>
                <div class="detail-item">
                    <span class="label">Current Balance:</span>
                    <span class="value">UGX ${Number(loan.balance).toLocaleString()}</span>
                </div>
                <div class="detail-item">
                    <span class="label">Interest Rate:</span>
                    <span class="value">${loan.interest_rate}%</span>
                </div>
                <div class="detail-item">
                    <span class="label">Status:</span>
                    <span class="value"><span class="status-badge status-${loan.status}">${loan.status}</span></span>
                </div>
                <div class="detail-item">
                    <span class="label">Repayment Mode:</span>
                    <span class="value">${loan.repayment_mode}</span>
                </div>
                ${loan.status === 'approved' ? `
                    <div style="background: #d4edda; border: 1px solid #c3e6cb; border-radius: 8px; padding: 15px; margin-top: 15px;">
                        <strong style="color: #155724;">✅ Congratulations!</strong>
                        <p style="margin: 10px 0 0 0; color: #155724;">Your loan has been approved! The accountant will disburse the funds shortly. You will be notified once the money has been disbursed to your account.</p>
                    </div>
                ` : ''}
                ${loan.status === 'pending' ? `
                    <div style="background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 8px; padding: 15px; margin-top: 15px;">
                        <strong style="color: #856404;">⏳ Under Review</strong>
                        <p style="margin: 10px 0 0 0; color: #856404;">Your loan application is currently being reviewed by your credit officer and manager.</p>
                    </div>
                ` : ''}
            `;
            
            // Show/hide top-up button based on eligibility
            if (topupButton) {
                if (response.eligibleForTopup) {
                    topupButton.style.display = 'inline-block';
                    topupButton.onclick = showLoanTopupModal;
                } else {
                    topupButton.style.display = 'none';
                    // Add info message about top-up eligibility
                    const eligibilityMsg = document.createElement('div');
                    eligibilityMsg.className = 'info-message';
                    eligibilityMsg.style.cssText = 'background: #e3f2fd; padding: 15px; border-radius: 8px; margin-top: 20px; border-left: 4px solid #2196F3;';
                    
                    const repaymentPct = parseFloat(response.repaymentPercentage || 0);
                    eligibilityMsg.innerHTML = `
                        <p><strong>ℹ️ Loan Top-up Information</strong></p>
                        <p>You have repaid ${repaymentPct.toFixed(1)}% of your current loan. 
                        Top-up loans become available after you've repaid at least 80% of your current loan.</p>
                        <p style="margin-top: 10px;"><strong>Remaining to qualify:</strong> ${Math.max(0, 80 - repaymentPct).toFixed(1)}%</p>
                    `;
                    content.appendChild(eligibilityMsg);
                }
            }
        }
    } catch (error) {
        console.error('Error loading loan overview:', error);
    }
}

async function loadRepaymentProgress() {
    try {
        const response = await apiCall('/client/repayment-progress');
        if (response.success && response.data) {
            const data = response.data;
            
            document.getElementById('clientProgressBar').style.width = `${data.progressPercentage}%`;
            document.getElementById('clientProgressText').textContent = `${data.progressPercentage}%`;
            
            // Format next payment date nicely
            const nextPaymentDateElem = document.getElementById('nextPaymentDate');
            if (data.loan.next_payment_date) {
                const nextDate = new Date(data.loan.next_payment_date);
                const today = new Date();
                const daysUntil = Math.ceil((nextDate - today) / (1000 * 60 * 60 * 24));
                
                const formattedDate = nextDate.toLocaleDateString('en-GB', { 
                    day: '2-digit', 
                    month: 'short', 
                    year: 'numeric' 
                });
                
                if (daysUntil < 0) {
                    nextPaymentDateElem.innerHTML = `<span style="color: red; font-weight: bold;">${formattedDate} (Overdue by ${Math.abs(daysUntil)} days)</span>`;
                } else if (daysUntil === 0) {
                    nextPaymentDateElem.innerHTML = `<span style="color: orange; font-weight: bold;">${formattedDate} (Due Today!)</span>`;
                } else if (daysUntil <= 3) {
                    nextPaymentDateElem.innerHTML = `<span style="color: orange;">${formattedDate} (${daysUntil} days)</span>`;
                } else {
                    nextPaymentDateElem.innerHTML = `${formattedDate} (in ${daysUntil} days)`;
                }
            } else {
                nextPaymentDateElem.textContent = 'N/A';
            }
            
            document.getElementById('nextPaymentAmount').textContent = `UGX ${Number(data.nextPaymentAmount).toLocaleString()}`;
            document.getElementById('penaltyAmount').textContent = `UGX ${Number(data.loan.penalty).toLocaleString()}`;
            
            // Show repayment history
            if (data.repaymentHistory.length > 0) {
                const historyHtml = `
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Amount</th>
                                <th>Method</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${data.repaymentHistory.map(payment => `
                                <tr>
                                    <td>${new Date(payment.payment_date).toLocaleDateString()}</td>
                                    <td>UGX ${Number(payment.amount).toLocaleString()}</td>
                                    <td>${payment.payment_method || 'N/A'}</td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                `;
                document.getElementById('repaymentHistoryTable').innerHTML = historyHtml;
            }
        }
    } catch (error) {
        console.error('Error loading repayment progress:', error);
    }
}

async function loadPortfolioGrowth() {
    try {
        const response = await apiCall('/client/portfolio-growth');
        if (response.success) {
            const data = response.data;
            
            // Update main stats
            document.getElementById('eligibleLimit').textContent = 
                `UGX ${Number(data.eligible_limit).toLocaleString()}`;
            document.getElementById('growthRate').textContent = `${data.growth_rate}%`;
            document.getElementById('growthMessage').textContent = data.growth_message;
            
            // Update progress bar
            document.getElementById('progressPercent').textContent = 
                `${data.progress_percent.toFixed(0)}%`;
            document.getElementById('progressBar').style.width = 
                `${data.progress_percent}%`;
            document.getElementById('nextMilestone').textContent = 
                data.next_milestone || 'Keep up the great work!';
            
            // Show/hide top-up badge
            const topupBadge = document.getElementById('topupBadge');
            if (data.qualifies_for_topup) {
                topupBadge.style.display = 'block';
            } else {
                topupBadge.style.display = 'none';
            }
            
            // Update dashboard stats bar progress
            document.getElementById('portfolioGrowth').style.width = `${data.growth_rate}%`;
        }
    } catch (error) {
        console.error('Error loading portfolio growth:', error);
    }
}

async function loadClientNotifications() {
    try {
        const response = await apiCall('/client/notifications');
        if (response.success) {
            const container = document.getElementById('clientNotificationsList');
            container.innerHTML = '';
            
            if (response.data.length === 0) {
                container.innerHTML = '<p class="text-center">No notifications</p>';
                return;
            }
            
            response.data.forEach(notif => {
                const div = document.createElement('div');
                div.className = `notification-item ${notif.type}`;
                div.innerHTML = `
                    <p>${notif.message}</p>
                    <small>${new Date(notif.created_at).toLocaleString()}</small>
                `;
                container.appendChild(div);
            });
        }
    } catch (error) {
        console.error('Error loading notifications:', error);
    }
}

function showClientSection(section) {
    document.querySelectorAll('#clientDashboard .dashboard-section').forEach(el => {
        el.classList.remove('active');
    });
    
    document.querySelectorAll('#clientDashboard .sidebar-link').forEach(link => {
        link.classList.remove('active');
    });
    
    document.getElementById(`client${section.charAt(0).toUpperCase() + section.slice(1)}`).classList.add('active');
    event.target.classList.add('active');
    
    // Load data for specific sections
    if (section === 'apply') {
        loadClientLoanApplicationStatus();
    }
}

// Load client loan application status
async function loadClientLoanApplicationStatus() {
    try {
        // First check application fee status
        const feeResponse = await apiCall('/client/application-fee-status');
        
        const applicationFeeSection = document.getElementById('applicationFeeSection');
        const pendingFeeAlert = document.getElementById('pendingFeeAlert');
        const existingAlert = document.getElementById('existingApplicationAlert');
        const applicationForm = document.getElementById('loanApplicationForm');
        
        // Check if client has paid and confirmed application fee
        if (!feeResponse.success || !feeResponse.hasConfirmedFee) {
            // Check if there's a pending fee
            if (feeResponse.hasPendingFee && feeResponse.pendingFee) {
                // Fee paid but waiting for confirmation
                applicationFeeSection.style.display = 'none';
                pendingFeeAlert.style.display = 'block';
                existingAlert.style.display = 'none';
                applicationForm.style.display = 'none';
                
                // Display pending fee details
                document.getElementById('pendingFeeAmountDisplay').textContent = `UGX ${Number(feeResponse.pendingFee.amount).toLocaleString()}`;
                document.getElementById('pendingFeeMethodDisplay').textContent = feeResponse.pendingFee.payment_method || 'N/A';
                document.getElementById('pendingFeeDateDisplay').textContent = new Date(feeResponse.pendingFee.created_at).toLocaleDateString();
                return;
            } else {
                // Client hasn't paid application fee yet
                applicationFeeSection.style.display = 'block';
                pendingFeeAlert.style.display = 'none';
                existingAlert.style.display = 'none';
                applicationForm.style.display = 'none';
                return;
            }
        }
        
        // Fee is confirmed, check if they have existing loan application
        const response = await apiCall('/client/loan-overview');
        
        const applicationDetails = document.getElementById('applicationDetails');
        
        if (response.success && response.data) {
            const loan = response.data;
            
            // Check if there's a pending or active loan
            if (['pending', 'approved', 'disbursed', 'active', 'overdue'].includes(loan.status)) {
                applicationFeeSection.style.display = 'none';
                pendingFeeAlert.style.display = 'none';
                existingAlert.style.display = 'block';
                applicationForm.style.display = 'none';
                
                let statusMessage = '';
                let statusColor = '#ffc107'; // yellow for pending
                
                if (loan.status === 'pending') {
                    statusMessage = '⏳ Pending Review';
                    statusColor = '#ffc107';
                } else if (loan.status === 'approved') {
                    statusMessage = '✅ Approved - Awaiting Disbursement';
                    statusColor = '#28a745';
                } else if (['disbursed', 'active'].includes(loan.status)) {
                    statusMessage = '💰 Active Loan';
                    statusColor = '#17a2b8';
                } else if (loan.status === 'overdue') {
                    statusMessage = '⚠️ Overdue';
                    statusColor = '#dc3545';
                }
                
                applicationDetails.innerHTML = `
                    <div style="background: white; padding: 15px; border-radius: 5px; border-left: 4px solid ${statusColor};">
                        <div style="margin-bottom: 8px;">
                            <strong>Status:</strong> <span style="color: ${statusColor}; font-weight: bold;">${statusMessage}</span>
                        </div>
                        <div style="margin-bottom: 8px;">
                            <strong>Loan Type:</strong> ${loan.loan_type}
                        </div>
                        <div style="margin-bottom: 8px;">
                            <strong>Amount Requested:</strong> UGX ${Number(loan.amount).toLocaleString()}
                        </div>
                        <div style="margin-bottom: 8px;">
                            <strong>Application Date:</strong> ${new Date(loan.created_at).toLocaleDateString()}
                        </div>
                        ${loan.status === 'pending' ? '<p style="margin-top: 10px; color: #666;">Your application is being reviewed by your credit officer and manager.</p>' : ''}
                        ${loan.status === 'approved' ? '<p style="margin-top: 10px; color: #28a745;">✅ Great news! Your loan has been approved and is being prepared for disbursement by the accountant.</p>' : ''}
                        ${['disbursed', 'active'].includes(loan.status) ? '<p style="margin-top: 10px; color: #17a2b8;">Your loan has been disbursed. Check the "Current Loan" section for full details and repayment information.</p>' : ''}
                    </div>
                `;
            } else {
                // No active loan, show the form (fee is already confirmed)
                applicationFeeSection.style.display = 'none';
                pendingFeeAlert.style.display = 'none';
                existingAlert.style.display = 'none';
                applicationForm.style.display = 'block';
            }
        } else {
            // New client with confirmed fee, show the form
            applicationFeeSection.style.display = 'none';
            pendingFeeAlert.style.display = 'none';
            existingAlert.style.display = 'none';
            applicationForm.style.display = 'block';
        }
    } catch (error) {
        console.error('Error checking loan status:', error);
    }
}

// Pay application fee (client confirms payment)
async function payApplicationFee() {
    const paymentMethod = document.getElementById('feePaymentMethod').value;
    const notes = document.getElementById('feeNotes').value;
    
    if (!paymentMethod) {
        alert('Please select a payment method');
        return;
    }
    
    if (!confirm(`Confirm that you have paid the application fee via ${paymentMethod}?\n\nThe accountant will verify your payment before you can apply for a loan.`)) {
        return;
    }
    
    try {
        const response = await apiCall('/client/pay-application-fee', 'POST', {
            amount: 5000, // Standard application fee
            payment_method: paymentMethod,
            transaction_reference: notes || null
        });
        
        if (response.success) {
            alert(`✅ ${response.message}\n\nPlease wait for accountant confirmation before applying for a loan.`);
            loadClientLoanApplicationStatus();
        } else {
            alert(`❌ ${response.message}`);
        }
    } catch (error) {
        console.error('Error paying application fee:', error);
        alert('Failed to record application fee payment. Please try again.');
    }
}

// Calculate new loan amount as user types
function calculateNewLoanAmount() {
    const currentLoan = window.currentClientLoan || {};
    const topupAmount = parseFloat(document.getElementById('topupAmount').value) || 0;
    const currentBalance = parseFloat(currentLoan.balance) || 0;
    const newLoanAmount = currentBalance + topupAmount;
    
    document.getElementById('newLoanAmount').textContent = 
        `UGX ${Number(newLoanAmount).toLocaleString()}`;
}

// ============================================
// ACCOUNTANT DASHBOARD FUNCTIONS
// ============================================

async function loadAccountantDashboard() {
    await loadFinancialSummary();
    await loadTransactions();
    await loadAccountantNotifications();
}

async function loadFinancialSummary() {
    try {
        const response = await apiCall('/accountant/summary');
        if (response.success) {
            const data = response.data;
            
            document.getElementById('totalFundsAvailable').textContent = 
                `UGX ${Number(data.totalFundsAvailable).toLocaleString()}`;
            document.getElementById('totalDisbursements').textContent = 
                `UGX ${Number(data.totalDisbursements).toLocaleString()}`;
            document.getElementById('totalDeposits').textContent = 
                `UGX ${Number(data.totalDeposits).toLocaleString()}`;
            document.getElementById('totalExpenditures').textContent = 
                `UGX ${Number(data.totalExpenditures).toLocaleString()}`;
            document.getElementById('totalInterestEarned').textContent = 
                `UGX ${Number(data.totalInterestEarned || 0).toLocaleString()}`;
            document.getElementById('completedLoansCount').textContent = 
                `From ${data.completedLoans || 0} completed loans`;
            
            // Update petty cash if element exists
            const pettyCashElement = document.getElementById('totalPettyCash');
            if (pettyCashElement) {
                pettyCashElement.textContent = `UGX ${Number(data.totalPettyCash || 0).toLocaleString()}`;
            }
        }
    } catch (error) {
        console.error('Error loading financial summary:', error);
    }
}

async function loadTransactions() {
    try {
        const response = await apiCall('/accountant/transactions');
        if (response.success) {
            const tbody = document.getElementById('transactionsTable');
            tbody.innerHTML = '';
            
            if (response.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="6" class="text-center">No transactions</td></tr>';
                return;
            }
            
            response.data.forEach(trans => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${trans.transaction_type}</td>
                    <td>UGX ${Number(trans.amount).toLocaleString()}</td>
                    <td>${trans.description || 'N/A'}</td>
                    <td>${new Date(trans.transaction_date).toLocaleDateString()}</td>
                    <td>${trans.created_by_name || 'System'}</td>
                    <td>
                        <button class="btn btn-sm btn-danger" onclick="deleteTransaction(${trans.id}, '${trans.transaction_type}')">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading transactions:', error);
    }
}

async function deleteTransaction(transactionId, transactionType) {
    if (!confirm(`Are you sure you want to delete this ${transactionType} transaction?\n\nThis action cannot be undone.`)) {
        return;
    }
    
    try {
        const response = await apiCall(`/accountant/transactions/${transactionId}`, 'DELETE');
        if (response.success) {
            alert('Transaction deleted successfully');
            loadTransactions();
        } else {
            alert(response.message || 'Failed to delete transaction');
        }
    } catch (error) {
        console.error('Error deleting transaction:', error);
        alert('Error deleting transaction: ' + error.message);
    }
}

async function loadAccountantNotifications() {
    try {
        const response = await apiCall('/accountant/notifications');
        if (response.success) {
            const container = document.getElementById('accountantNotificationsList');
            container.innerHTML = '';
            
            if (response.data.lowFunds) {
                const alert = document.createElement('div');
                alert.className = 'notification-item alert';
                alert.innerHTML = `
                    <strong>Low Funds Alert!</strong>
                    <p>Available funds: UGX ${Number(response.data.fundsAvailable).toLocaleString()}</p>
                `;
                container.appendChild(alert);
            }
            
            if (response.data.pendingDisbursements > 0) {
                const alert = document.createElement('div');
                alert.className = 'notification-item warning';
                alert.innerHTML = `
                    <strong>Pending Disbursements:</strong>
                    <p>${response.data.pendingDisbursements} loan(s) approved and awaiting disbursement</p>
                `;
                container.appendChild(alert);
            }
        }
    } catch (error) {
        console.error('Error loading accountant notifications:', error);
    }
}

function showAccountantSection(section) {
    document.querySelectorAll('#accountantDashboard .dashboard-section').forEach(el => {
        el.classList.remove('active');
        el.style.display = 'none';
    });
    
    document.querySelectorAll('#accountantDashboard .sidebar-link').forEach(link => {
        link.classList.remove('active');
    });
    
    // Support kebab-case section names (e.g. "petty-cash") by converting to CamelCase
    const toCamel = s => s.split('-').map(p => p.charAt(0).toUpperCase() + p.slice(1)).join('');
    const targetId = `accountant${toCamel(section)}`;
    const targetSection = document.getElementById(targetId);
    if (!targetSection) {
        console.warn('Accountant section not found:', targetId);
        return;
    }
    targetSection.classList.add('active');
    targetSection.style.display = 'block';
    
    if (event && event.target) {
        event.target.classList.add('active');
    }
    
    // Load data for specific sections
    if (section === 'petty-cash' || section === 'pettycash') {
        loadPettyCash();
    } else if (section === 'fees') {
        loadPendingApplicationFees();
    } else if (section === 'disbursements') {
        loadApprovedLoansForDisbursement();
    } else if (section === 'salaries') {
        loadStaffForSalary();
        loadSalaryHistory();
    } else if (section === 'payments') {
        // Record Payments section - focus search input for user convenience
        setTimeout(() => {
            const searchInput = document.getElementById('searchClientPayment');
            if (searchInput) {
                searchInput.focus();
            }
        }, 100);
    }
}

async function showInterestDetails() {
    // Hide all sections
    document.querySelectorAll('#accountantDashboard .dashboard-section').forEach(el => {
        el.classList.remove('active');
        el.style.display = 'none';
    });
    
    // Show interest section
    const interestSection = document.getElementById('accountantInterest');
    interestSection.classList.add('active');
    interestSection.style.display = 'block';
    
    // Load interest data
    try {
        const response = await apiCall('/accountant/interest-earned');
        if (response.success) {
            const { loans, summary } = response.data;
            
            // Update summary cards
            const summaryCards = document.getElementById('interestSummaryCards');
            summaryCards.innerHTML = `
                <div class="card">
                    <div class="card-icon">🎯</div>
                    <div class="card-content">
                        <h3>${summary.totalLoans}</h3>
                        <p>Completed Loans</p>
                    </div>
                </div>
                <div class="card">
                    <div class="card-icon">💵</div>
                    <div class="card-content">
                        <h3>UGX ${Number(summary.totalPrincipal).toLocaleString()}</h3>
                        <p>Total Principal</p>
                    </div>
                </div>
                <div class="card">
                    <div class="card-icon">💹</div>
                    <div class="card-content">
                        <h3>UGX ${Number(summary.totalInterest).toLocaleString()}</h3>
                        <p>Total Interest Earned</p>
                    </div>
                </div>
                <div class="card">
                    <div class="card-icon">💰</div>
                    <div class="card-content">
                        <h3>UGX ${Number(summary.totalCollected).toLocaleString()}</h3>
                        <p>Total Collected</p>
                    </div>
                </div>
            `;
            
            // Update table
            const tbody = document.querySelector('#interestDetailsTable tbody');
            tbody.innerHTML = '';
            
            if (loans.length === 0) {
                tbody.innerHTML = '<tr><td colspan="9" class="text-center">No completed loans yet</td></tr>';
                return;
            }
            
            loans.forEach(loan => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${loan.loan_number}</td>
                    <td>${loan.client_name}</td>
                    <td>UGX ${Number(loan.principal).toLocaleString()}</td>
                    <td>${loan.interest_rate}%</td>
                    <td>${loan.duration_months} months</td>
                    <td><strong>UGX ${Number(loan.interest_earned).toLocaleString()}</strong></td>
                    <td>UGX ${Number(loan.total_paid).toLocaleString()}</td>
                    <td>${new Date(loan.disbursed_at).toLocaleDateString()}</td>
                    <td>${new Date(loan.completed_at).toLocaleDateString()}</td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading interest details:', error);
        alert('Failed to load interest details');
    }
}

async function loadPettyCash() {
    try {
        const response = await apiCall('/accountant/petty-cash');
        if (response.success) {
            // Update balance display
            if (response.balance) {
                document.getElementById('pettyCashAllocated').textContent = 
                    `UGX ${Number(response.balance.allocated).toLocaleString()}`;
                document.getElementById('pettyCashBalance').textContent = 
                    `UGX ${Number(response.balance.current).toLocaleString()}`;
                document.getElementById('pettyCashSpent').textContent = 
                    `UGX ${Number(response.balance.spent).toLocaleString()}`;
            }
            
            // Update table
            const tbody = document.getElementById('pettyCashTable');
            tbody.innerHTML = '';
            
            if (response.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="6" class="text-center">No petty cash records</td></tr>';
                return;
            }
            
            response.data.forEach(cash => {
                const row = document.createElement('tr');
                const type = cash.transaction_type === 'allocation' ? '💵 Allocation' : 
                             cash.transaction_type === 'top-up' ? '➕ Top-up' : '📝 Expense';
                row.innerHTML = `
                    <td>${type}</td>
                    <td>UGX ${Number(cash.amount).toLocaleString()}</td>
                    <td>${cash.description}</td>
                    <td>${new Date(cash.transaction_date).toLocaleDateString()}</td>
                    <td>${cash.recorded_by_name || 'N/A'}</td>
                    <td>
                        <button class="btn btn-sm btn-danger" onclick="deletePettyCashTransaction(${cash.id}, '${cash.transaction_type}')">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading petty cash:', error);
    }
}

// =============== SALARY MANAGEMENT FUNCTIONS ===============

// Load staff list for salary payment
async function loadStaffForSalary() {
    try {
        const response = await apiCall('/accountant/staff-list');
        if (response.success) {
            const select = document.getElementById('salaryStaffId');
            select.innerHTML = '<option value="">-- Select Staff --</option>';
            
            response.data.forEach(staff => {
                const option = document.createElement('option');
                option.value = staff.id;
                option.textContent = `${staff.name} (${staff.role})`;
                option.dataset.role = staff.role;
                select.appendChild(option);
            });
        }
    } catch (error) {
        console.error('Error loading staff list:', error);
    }
}

// Load salary payment history
async function loadSalaryHistory() {
    try {
        const response = await apiCall('/accountant/salary-history');
        if (response.success) {
            const tbody = document.getElementById('salaryHistoryTable');
            tbody.innerHTML = '';
            
            if (response.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="10" class="text-center">No salary payments recorded</td></tr>';
                return;
            }
            
            response.data.forEach(salary => {
                const row = document.createElement('tr');
                const netAmount = salary.base_salary + salary.bonuses - salary.deductions;
                const paymentDate = new Date(salary.payment_date).toLocaleDateString();
                
                row.innerHTML = `
                    <td>${paymentDate}</td>
                    <td>${salary.staff_name}</td>
                    <td>${salary.staff_role}</td>
                    <td>${salary.salary_period}</td>
                    <td>UGX ${Number(salary.base_salary).toLocaleString()}</td>
                    <td style="color: green;">+${Number(salary.bonuses).toLocaleString()}</td>
                    <td style="color: red;">-${Number(salary.deductions).toLocaleString()}</td>
                    <td><strong>UGX ${netAmount.toLocaleString()}</strong></td>
                    <td>${salary.payment_method}</td>
                    <td>${salary.reference || '-'}</td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading salary history:', error);
    }
}

// Pay staff salary
async function paySalary(event) {
    event.preventDefault();
    
    const staffId = document.getElementById('salaryStaffId').value;
    const period = document.getElementById('salaryPeriod').value;
    const baseSalary = Number(document.getElementById('baseSalary').value);
    const bonuses = Number(document.getElementById('salaryBonuses').value) || 0;
    const deductions = Number(document.getElementById('salaryDeductions').value) || 0;
    const paymentMethod = document.getElementById('salaryPaymentMethod').value;
    const reference = document.getElementById('salaryReference').value;
    const notes = document.getElementById('salaryNotes').value;
    
    if (!staffId || !period || !baseSalary) {
        alert('Please fill in all required fields');
        return;
    }
    
    const netAmount = baseSalary + bonuses - deductions;
    
    if (netAmount <= 0) {
        alert('Net salary amount must be greater than zero');
        return;
    }
    
    if (!confirm(`Pay salary of UGX ${netAmount.toLocaleString()} to this staff member for ${period}?`)) {
        return;
    }
    
    try {
        const response = await apiCall('/accountant/pay-salary', 'POST', {
            staff_id: staffId,
            salary_period: period,
            base_salary: baseSalary,
            bonuses: bonuses,
            deductions: deductions,
            payment_method: paymentMethod,
            reference: reference,
            notes: notes
        });
        
        if (response.success) {
            alert('✅ Salary paid successfully!');
            document.getElementById('paySalaryForm').reset();
            loadSalaryHistory();
        } else {
            alert(response.message || 'Failed to pay salary');
        }
    } catch (error) {
        console.error('Error paying salary:', error);
        alert('Error paying salary: ' + error.message);
    }
}

// =============== END SALARY MANAGEMENT ===============

// =============== RECEIPT SEARCH & PRINT ===============

let currentReceiptNumber = null;

async function searchReceipt() {
    const receiptNumber = document.getElementById('searchReceiptNumber').value.trim();
    
    if (!receiptNumber) {
        alert('Please enter a receipt number');
        return;
    }
    
    try {
        const response = await apiCall(`/accountant/receipt/search/${receiptNumber}`);
        
        if (response.success) {
            currentReceiptNumber = receiptNumber;
            displayReceiptDetails(response.payment);
            document.getElementById('receiptSearchResult').style.display = 'block';
        } else {
            alert(response.message || 'Receipt not found');
            document.getElementById('receiptSearchResult').style.display = 'none';
        }
    } catch (error) {
        console.error('Error searching receipt:', error);
        alert('Error searching receipt: ' + error.message);
    }
}

function displayReceiptDetails(payment) {
    const paymentDate = new Date(payment.payment_date);
    const formattedDate = paymentDate.toLocaleDateString('en-GB');
    const formattedTime = paymentDate.toLocaleTimeString('en-GB');
    
    const detailsHtml = `
        <div style="border-bottom: 2px solid #2563eb; padding-bottom: 15px; margin-bottom: 15px;">
            <h3 style="color: #2563eb; margin: 0;">Receipt Number: ${payment.receipt_number}</h3>
            <p style="color: #666; margin: 5px 0 0 0;">Date: ${formattedDate} | Time: ${formattedTime}</p>
        </div>
        
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 20px;">
            <div>
                <h4 style="margin: 0 0 10px 0; color: #333;">Client Information</h4>
                <p><strong>Name:</strong> ${payment.client_name}</p>
                <p><strong>Phone:</strong> ${payment.client_phone}</p>
                ${payment.client_email ? `<p><strong>Email:</strong> ${payment.client_email}</p>` : ''}
            </div>
            <div>
                <h4 style="margin: 0 0 10px 0; color: #333;">Loan Details</h4>
                <p><strong>Loan ID:</strong> #${payment.loan_id}</p>
                <p><strong>Loan Type:</strong> ${payment.loan_type}</p>
                <p><strong>Payment Method:</strong> ${payment.payment_method.replace('_', ' ').toUpperCase()}</p>
            </div>
        </div>
        
        <div style="background: #f0f9ff; padding: 20px; border-radius: 8px; text-align: center; margin-bottom: 15px;">
            <h4 style="margin: 0 0 10px 0; color: #666;">AMOUNT PAID</h4>
            <h2 style="margin: 0; color: #2563eb; font-size: 32px;">UGX ${Number(payment.amount).toLocaleString()}</h2>
        </div>
        
        <div>
            <p><strong>Recorded By:</strong> ${payment.recorded_by_name || 'System'}</p>
            ${payment.notes ? `<p><strong>Notes:</strong> ${payment.notes}</p>` : ''}
        </div>
    `;
    
    document.getElementById('receiptDetailsCard').innerHTML = detailsHtml;
}

function printReceipt() {
    if (!currentReceiptNumber) {
        alert('No receipt selected');
        return;
    }
    
    window.open(`/api/accountant/receipt/print/${currentReceiptNumber}`, '_blank');
}

// =============== END RECEIPT SEARCH & PRINT ===============

async function deletePettyCashTransaction(transactionId, transactionType) {
    if (!confirm(`Are you sure you want to delete this ${transactionType} petty cash transaction?\n\nThis will affect the petty cash balance. This action cannot be undone.`)) {
        return;
    }
    
    try {
        const response = await apiCall(`/accountant/petty-cash/${transactionId}`, 'DELETE');
        if (response.success) {
            alert('Petty cash transaction deleted successfully');
            loadPettyCash();
        } else {
            alert(response.message || 'Failed to delete petty cash transaction');
        }
    } catch (error) {
        console.error('Error deleting petty cash transaction:', error);
        alert('Error deleting petty cash transaction: ' + error.message);
    }
}

async function topUpAccountFunds(accountCode, accountName) {
    if (!confirm(`This will top up ${accountName} to the full allocated amount. Continue?`)) {
        return;
    }
    
    try {
        const response = await apiCall('/accountant/account-allocations/top-up', 'POST', {
            account_code: accountCode,
            description: `${accountName} replenishment`
        });
        
        if (response.success) {
            alert(`✅ ${response.message}`);
            loadAccountBalances(); // Reload balances and allocations
            loadFinancialSummary(); // Reload financial summary
        }
    } catch (error) {
        alert('❌ Failed to top up account. ' + (error.message || 'Please try again.'));
    }
}

async function topUpPettyCash() {
    if (!confirm('This will top up petty cash to the full allocated amount. Continue?')) {
        return;
    }
    
    try {
        const response = await apiCall('/accountant/petty-cash/top-up', 'POST', {
            description: 'Balance replenishment'
        });
        
        if (response.success) {
            alert(`✅ ${response.message}`);
            loadPettyCash(); // Reload petty cash data
            loadFinancialSummary(); // Reload financial summary
        }
    } catch (error) {
        alert('❌ Failed to top up petty cash. ' + (error.message || 'Please try again.'));
    }
}

// Show payment modal for accountant
function showAccountantRecordPaymentModal(loanId, clientName, balance) {
    document.getElementById('paymentLoanId').value = loanId;
    document.getElementById('paymentClientName').textContent = clientName;
    document.getElementById('paymentLoanBalance').textContent = `UGX ${Number(balance).toLocaleString()}`;
    document.getElementById('paymentAmount').setAttribute('max', balance);
    document.getElementById('paymentAmount').value = '';
    document.getElementById('paymentMethodSelect').value = '';
    document.getElementById('receiptNumber').value = '';
    document.getElementById('paymentNotes').value = '';
    
    openModal('recordPaymentModal');
}

function showAddTransactionModal() {
    openModal('addTransactionModal');
}

function showAddPettyCashModal() {
    const userRole = localStorage.getItem('userRole');
    
    // Managers can only allocate, not record expenses
    if (userRole === 'manager') {
        openModal('allocatePettyCashModal');
    } else {
        // Accountants can record expenses
        openModal('addPettyCashModal');
    }
}

function showAddFundsModal() {
    openModal('addFundsModal');
}

async function generateAccountantReport() {
    const startDate = document.getElementById('reportStartDate').value;
    const endDate = document.getElementById('reportEndDate').value;
    
    if (!startDate || !endDate) {
        alert('Please select start and end dates');
        return;
    }
    
    try {
        const response = await apiCall(`/accountant/report?start_date=${startDate}&end_date=${endDate}`);
        if (response.success) {
            const container = document.getElementById('accountantReportResults');
            container.innerHTML = `
                <h3>Financial Report</h3>
                <p><strong>Period:</strong> ${startDate} to ${endDate}</p>
                
                <h4>Transaction Summary</h4>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Type</th>
                            <th>Count</th>
                            <th>Total Amount</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${response.data.transactions.map(t => `
                            <tr>
                                <td>${t.transaction_type}</td>
                                <td>${t.count}</td>
                                <td>UGX ${Number(t.total_amount).toLocaleString()}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
                
                <h4>Loan Statistics</h4>
                <div class="summary-cards">
                    <div class="card">
                        <div class="card-content">
                            <h3>${response.data.loanStats.total_loans}</h3>
                            <p>Total Loans</p>
                        </div>
                    </div>
                    <div class="card">
                        <div class="card-content">
                            <h3>UGX ${Number(response.data.loanStats.total_loaned).toLocaleString()}</h3>
                            <p>Total Loaned</p>
                        </div>
                    </div>
                    <div class="card">
                        <div class="card-content">
                            <h3>UGX ${Number(response.data.loanStats.total_collected).toLocaleString()}</h3>
                            <p>Total Collected</p>
                        </div>
                    </div>
                    <div class="card">
                        <div class="card-content">
                            <h3>UGX ${Number(response.data.loanStats.total_outstanding).toLocaleString()}</h3>
                            <p>Outstanding</p>
                        </div>
                    </div>
                </div>
            `;
        }
    } catch (error) {
        console.error('Error generating report:', error);
        alert('Failed to generate report');
    }
}

// ============================================
// COMPREHENSIVE FINANCIAL REPORTS
// ============================================

async function generateFinancialReport() {
    const reportType = document.getElementById('reportType').value;
    const startDate = document.getElementById('reportStartDate').value;
    const endDate = document.getElementById('reportEndDate').value;
    
    if (!startDate || !endDate) {
        alert('⚠️ Please select both start and end dates');
        return;
    }
    
    if (new Date(startDate) > new Date(endDate)) {
        alert('⚠️ Start date must be before end date');
        return;
    }
    
    const reportDisplayArea = document.getElementById('reportDisplayArea');
    const reportContent = document.getElementById('reportContent');
    const reportPeriodDisplay = document.getElementById('reportPeriodDisplay');
    const reportGeneratedDate = document.getElementById('reportGeneratedDate');
    
    // Show loading
    reportContent.innerHTML = '<div class="text-center"><p>⏳ Generating report...</p></div>';
    reportDisplayArea.style.display = 'block';
    
    // Set report metadata
    reportPeriodDisplay.textContent = `${new Date(startDate).toLocaleDateString()} to ${new Date(endDate).toLocaleDateString()}`;
    reportGeneratedDate.textContent = new Date().toLocaleString();
    
    try {
        let reportHTML = '';
        
        switch(reportType) {
            case 'income-statement':
                reportHTML = await generateIncomeStatementHTML(startDate, endDate);
                break;
            case 'balance-sheet':
                reportHTML = await generateBalanceSheetHTML(startDate, endDate);
                break;
            case 'cash-flow':
                reportHTML = await generateCashFlowHTML(startDate, endDate);
                break;
            case 'loan-portfolio':
                reportHTML = await generateLoanPortfolioHTML(startDate, endDate);
                break;
            case 'collections-report':
                reportHTML = await generateCollectionsHTML(startDate, endDate);
                break;
            case 'disbursements-report':
                reportHTML = await generateDisbursementsHTML(startDate, endDate);
                break;
            case 'performance-report':
                reportHTML = await generatePerformanceHTML(startDate, endDate);
                break;
            default:
                reportHTML = '<p>Invalid report type</p>';
        }
        
        reportContent.innerHTML = reportHTML;
        
        // Show print and export buttons
        document.getElementById('printReportBtn').style.display = 'inline-flex';
        document.getElementById('exportPDFBtn').style.display = 'inline-flex';
        
    } catch (error) {
        console.error('Error generating financial report:', error);
        reportContent.innerHTML = '<div class="message error">❌ Failed to generate report. Please try again.</div>';
    }
}

async function generateIncomeStatementHTML(startDate, endDate) {
    const response = await apiCall(`/financial-reports/income-statement?startDate=${startDate}&endDate=${endDate}`);
    
    if (!response.success) throw new Error('Failed to generate income statement');
    
    const data = response.data;
    
    return `
        <h2 class="report-title">INCOME STATEMENT (PROFIT & LOSS)</h2>
        
        <table class="financial-table">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: var(--light-blue);">REVENUE</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Loan Interest Income</td>
                    <td class="amount">UGX ${Number(data.revenue.loanInterest).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Processing & Form Fees</td>
                    <td class="amount">UGX ${Number(data.revenue.processingFees).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Penalty Fees</td>
                    <td class="amount">UGX ${Number(data.revenue.penaltyFees).toLocaleString()}</td>
                </tr>
                <tr class="total-row">
                    <td><strong>TOTAL REVENUE</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.revenue.total).toLocaleString()}</strong></td>
                </tr>
            </tbody>
        </table>
        
        <table class="financial-table mt-3">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: var(--light-blue);">EXPENSES</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Operating Expenses</td>
                    <td class="amount">UGX ${Number(data.expenses.operating).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Petty Cash Expenses</td>
                    <td class="amount">UGX ${Number(data.expenses.pettyCash).toLocaleString()}</td>
                </tr>
                <tr class="total-row">
                    <td><strong>TOTAL EXPENSES</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.expenses.total).toLocaleString()}</strong></td>
                </tr>
            </tbody>
        </table>
        
        <table class="financial-table mt-3">
            <tbody>
                <tr class="highlight-row ${data.netIncome >= 0 ? 'positive' : 'negative'}">
                    <td><strong>NET ${data.netIncome >= 0 ? 'PROFIT' : 'LOSS'}</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.netIncome).toLocaleString()}</strong></td>
                </tr>
                <tr>
                    <td>Profit Margin</td>
                    <td class="amount">${data.profitMargin}%</td>
                </tr>
            </tbody>
        </table>
    `;
}

async function generateBalanceSheetHTML(startDate, asOfDate) {
    const response = await apiCall(`/financial-reports/balance-sheet?asOfDate=${asOfDate}`);
    
    if (!response.success) throw new Error('Failed to generate balance sheet');
    
    const data = response.data;
    
    return `
        <h2 class="report-title">BALANCE SHEET</h2>
        <p class="report-subtitle">As of ${new Date(asOfDate).toLocaleDateString()}</p>
        
        <table class="financial-table">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: var(--light-blue);">ASSETS</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Cash on Hand</td>
                    <td class="amount">UGX ${Number(data.assets.cash).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Loans Receivable</td>
                    <td class="amount">UGX ${Number(data.assets.loansReceivable).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Interest Receivable</td>
                    <td class="amount">UGX ${Number(data.assets.interestReceivable).toLocaleString()}</td>
                </tr>
                <tr class="total-row">
                    <td><strong>TOTAL ASSETS</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.assets.total).toLocaleString()}</strong></td>
                </tr>
            </tbody>
        </table>
        
        <table class="financial-table mt-3">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: var(--light-green);">LIABILITIES</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Member Savings Deposits</td>
                    <td class="amount">UGX ${Number(data.liabilities.savingsDeposits).toLocaleString()}</td>
                </tr>
                <tr class="total-row">
                    <td><strong>TOTAL LIABILITIES</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.liabilities.total).toLocaleString()}</strong></td>
                </tr>
            </tbody>
        </table>
        
        <table class="financial-table mt-3">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: var(--light-green);">EQUITY</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Retained Earnings</td>
                    <td class="amount">UGX ${Number(data.equity.retainedEarnings).toLocaleString()}</td>
                </tr>
                <tr class="total-row">
                    <td><strong>TOTAL EQUITY</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.equity.total).toLocaleString()}</strong></td>
                </tr>
            </tbody>
        </table>
        
        <table class="financial-table mt-3">
            <tbody>
                <tr class="highlight-row">
                    <td><strong>TOTAL LIABILITIES & EQUITY</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.totalLiabilitiesAndEquity).toLocaleString()}</strong></td>
                </tr>
            </tbody>
        </table>
    `;
}

async function generateCashFlowHTML(startDate, endDate) {
    const response = await apiCall(`/financial-reports/cash-flow?startDate=${startDate}&endDate=${endDate}`);
    
    if (!response.success) throw new Error('Failed to generate cash flow statement');
    
    const data = response.data;
    
    return `
        <h2 class="report-title">CASH FLOW STATEMENT</h2>
        
        <table class="financial-table">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: var(--light-blue);">OPERATING ACTIVITIES</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Cash from Loan Repayments</td>
                    <td class="amount">UGX ${Number(data.operating.inflows).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Cash for Operating Expenses</td>
                    <td class="amount negative">(UGX ${Number(data.operating.outflows).toLocaleString()})</td>
                </tr>
                <tr class="total-row">
                    <td><strong>Net Cash from Operating</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.operating.net).toLocaleString()}</strong></td>
                </tr>
            </tbody>
        </table>
        
        <table class="financial-table mt-3">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: var(--light-green);">FINANCING ACTIVITIES</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Member Deposits</td>
                    <td class="amount">UGX ${Number(data.financing.inflows).toLocaleString()}</td>
                </tr>
                <tr class="total-row">
                    <td><strong>Net Cash from Financing</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.financing.net).toLocaleString()}</strong></td>
                </tr>
            </tbody>
        </table>
        
        <table class="financial-table mt-3">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: #fff3e0;">INVESTING ACTIVITIES</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Loan Disbursements</td>
                    <td class="amount negative">(UGX ${Number(data.investing.outflows).toLocaleString()})</td>
                </tr>
                <tr class="total-row">
                    <td><strong>Net Cash from Investing</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.investing.net).toLocaleString()}</strong></td>
                </tr>
            </tbody>
        </table>
        
        <table class="financial-table mt-3">
            <tbody>
                <tr class="highlight-row ${data.netCashFlow >= 0 ? 'positive' : 'negative'}">
                    <td><strong>NET CHANGE IN CASH</strong></td>
                    <td class="amount"><strong>UGX ${Number(data.netCashFlow).toLocaleString()}</strong></td>
                </tr>
            </tbody>
        </table>
    `;
}

async function generateLoanPortfolioHTML(startDate, endDate) {
    const response = await apiCall(`/financial-reports/loan-portfolio?startDate=${startDate}&endDate=${endDate}`);
    
    if (!response.success) throw new Error('Failed to generate loan portfolio report');
    
    const data = response.data;
    
    let loanTypeRows = data.loansByType.map(type => `
        <tr>
            <td>${type.loan_type}</td>
            <td>${type.count}</td>
            <td class="amount">UGX ${Number(type.total_amount).toLocaleString()}</td>
            <td class="amount">UGX ${Number(type.outstanding).toLocaleString()}</td>
        </tr>
    `).join('');
    
    return `
        <h2 class="report-title">LOAN PORTFOLIO REPORT</h2>
        
        <div class="summary-cards mb-3">
            <div class="card blue">
                <h3>Total Loans</h3>
                <div class="value">${data.summary.total_loans}</div>
            </div>
            <div class="card green">
                <h3>Total Disbursed</h3>
                <div class="value">UGX ${Number(data.summary.total_disbursed).toLocaleString()}</div>
            </div>
            <div class="card warning">
                <h3>Outstanding</h3>
                <div class="value">UGX ${Number(data.summary.total_outstanding).toLocaleString()}</div>
            </div>
            <div class="card">
                <h3>Repayment Rate</h3>
                <div class="value">${data.repaymentRate}%</div>
            </div>
        </div>
        
        <table class="financial-table">
            <thead>
                <tr>
                    <th>Status</th>
                    <th>Count</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Active Loans</td>
                    <td><span class="badge success">${data.summary.active_loans}</span></td>
                </tr>
                <tr>
                    <td>Completed Loans</td>
                    <td><span class="badge info">${data.summary.completed_loans}</span></td>
                </tr>
                <tr>
                    <td>Defaulted Loans</td>
                    <td><span class="badge error">${data.summary.defaulted_loans}</span></td>
                </tr>
            </tbody>
        </table>
        
        <h3 class="mt-3 mb-2">Portfolio by Loan Type</h3>
        <table class="financial-table">
            <thead>
                <tr>
                    <th>Loan Type</th>
                    <th>Count</th>
                    <th>Total Amount</th>
                    <th>Outstanding</th>
                </tr>
            </thead>
            <tbody>
                ${loanTypeRows || '<tr><td colspan="4" class="text-center">No data</td></tr>'}
            </tbody>
        </table>
    `;
}

async function generateCollectionsHTML(startDate, endDate) {
    const response = await apiCall(`/financial-reports/collections?startDate=${startDate}&endDate=${endDate}`);
    
    if (!response.success) throw new Error('Failed to generate collections report');
    
    const data = response.data;
    
    let collectionsRows = data.collections.slice(0, 50).map(col => `
        <tr>
            <td>${new Date(col.payment_date).toLocaleDateString()}</td>
            <td>${col.client_name}</td>
            <td>${col.loan_type}</td>
            <td class="amount">UGX ${Number(col.amount_paid).toLocaleString()}</td>
            <td class="amount">UGX ${Number(col.principal_paid).toLocaleString()}</td>
            <td class="amount">UGX ${Number(col.interest_paid).toLocaleString()}</td>
            <td>${col.payment_method || 'N/A'}</td>
        </tr>
    `).join('');
    
    return `
        <h2 class="report-title">COLLECTIONS REPORT</h2>
        
        <table class="financial-table mb-3">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: var(--light-blue);">COLLECTION SUMMARY</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Total Transactions</td>
                    <td class="amount">${data.summary.total_transactions}</td>
                </tr>
                <tr>
                    <td>Total Collected</td>
                    <td class="amount">UGX ${Number(data.summary.total_collected).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Principal Collected</td>
                    <td class="amount">UGX ${Number(data.summary.total_principal).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Interest Collected</td>
                    <td class="amount">UGX ${Number(data.summary.total_interest).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Penalties Collected</td>
                    <td class="amount">UGX ${Number(data.summary.total_penalties).toLocaleString()}</td>
                </tr>
            </tbody>
        </table>
        
        <h3 class="mb-2">Recent Collections (Last 50)</h3>
        <table class="financial-table">
            <thead>
                <tr>
                    <th>Date</th>
                    <th>Client</th>
                    <th>Loan Type</th>
                    <th>Amount Paid</th>
                    <th>Principal</th>
                    <th>Interest</th>
                    <th>Method</th>
                </tr>
            </thead>
            <tbody>
                ${collectionsRows || '<tr><td colspan="7" class="text-center">No collections in this period</td></tr>'}
            </tbody>
        </table>
    `;
}

async function generateDisbursementsHTML(startDate, endDate) {
    const response = await apiCall(`/financial-reports/disbursements?startDate=${startDate}&endDate=${endDate}`);
    
    if (!response.success) throw new Error('Failed to generate disbursements report');
    
    const data = response.data;
    
    let disbursementRows = data.disbursements.map(disb => `
        <tr>
            <td>${new Date(disb.disbursement_date).toLocaleDateString()}</td>
            <td>${disb.client_name}</td>
            <td>${disb.loan_type}</td>
            <td class="amount">UGX ${Number(disb.amount).toLocaleString()}</td>
            <td>${disb.interest_rate}%</td>
            <td>${disb.duration_months} months</td>
            <td>${disb.officer_name || 'N/A'}</td>
        </tr>
    `).join('');
    
    return `
        <h2 class="report-title">DISBURSEMENTS REPORT</h2>
        
        <table class="financial-table mb-3">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: var(--light-blue);">DISBURSEMENT SUMMARY</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Total Loans Disbursed</td>
                    <td class="amount">${data.summary.total_loans}</td>
                </tr>
                <tr>
                    <td>Total Amount Disbursed</td>
                    <td class="amount">UGX ${Number(data.summary.total_disbursed).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Processing Fees Collected</td>
                    <td class="amount">UGX ${Number(data.summary.total_processing_fees).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Form Fees Collected</td>
                    <td class="amount">UGX ${Number(data.summary.total_form_fees).toLocaleString()}</td>
                </tr>
            </tbody>
        </table>
        
        <h3 class="mb-2">Disbursement Details</h3>
        <table class="financial-table">
            <thead>
                <tr>
                    <th>Date</th>
                    <th>Client</th>
                    <th>Loan Type</th>
                    <th>Amount</th>
                    <th>Interest</th>
                    <th>Duration</th>
                    <th>Officer</th>
                </tr>
            </thead>
            <tbody>
                ${disbursementRows || '<tr><td colspan="7" class="text-center">No disbursements in this period</td></tr>'}
            </tbody>
        </table>
    `;
}

async function generatePerformanceHTML(startDate, endDate) {
    const response = await apiCall(`/financial-reports/performance?startDate=${startDate}&endDate=${endDate}`);
    
    if (!response.success) throw new Error('Failed to generate performance report');
    
    const data = response.data;
    
    let officerRows = data.officerPerformance.map(officer => `
        <tr>
            <td>${officer.officer_name}</td>
            <td>${officer.loans_managed}</td>
            <td class="amount">UGX ${Number(officer.total_disbursed).toLocaleString()}</td>
            <td class="amount">UGX ${Number(officer.total_collected).toLocaleString()}</td>
            <td>${officer.total_disbursed > 0 ? ((officer.total_collected / officer.total_disbursed) * 100).toFixed(1) : 0}%</td>
        </tr>
    `).join('');
    
    return `
        <h2 class="report-title">PERFORMANCE SUMMARY</h2>
        
        <div class="summary-cards mb-3">
            <div class="card blue">
                <h3>Total Clients</h3>
                <div class="value">${data.totalClients}</div>
            </div>
            <div class="card green">
                <h3>Disbursed</h3>
                <div class="value">UGX ${Number(data.portfolioGrowth.total_disbursed).toLocaleString()}</div>
            </div>
            <div class="card warning">
                <h3>Outstanding</h3>
                <div class="value">UGX ${Number(data.portfolioGrowth.total_outstanding).toLocaleString()}</div>
            </div>
            <div class="card ${parseFloat(data.defaultRate) > 5 ? 'error' : 'success'}">
                <h3>Default Rate</h3>
                <div class="value">${data.defaultRate}%</div>
            </div>
        </div>
        
        <table class="financial-table mb-3">
            <thead>
                <tr>
                    <th colspan="2" style="text-align: left; background: var(--light-blue);">COLLECTION PERFORMANCE</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Total Collections</td>
                    <td class="amount">UGX ${Number(data.repaymentPerformance.total_collected).toLocaleString()}</td>
                </tr>
                <tr>
                    <td>Number of Payments</td>
                    <td class="amount">${data.repaymentPerformance.total_payments}</td>
                </tr>
                <tr>
                    <td>Average Payment</td>
                    <td class="amount">UGX ${data.repaymentPerformance.total_payments > 0 ? Number(data.repaymentPerformance.total_collected / data.repaymentPerformance.total_payments).toLocaleString() : 0}</td>
                </tr>
            </tbody>
        </table>
        
        <h3 class="mb-2">Officer Performance</h3>
        <table class="financial-table">
            <thead>
                <tr>
                    <th>Officer Name</th>
                    <th>Loans Managed</th>
                    <th>Total Disbursed</th>
                    <th>Total Collected</th>
                    <th>Collection Rate</th>
                </tr>
            </thead>
            <tbody>
                ${officerRows || '<tr><td colspan="5" class="text-center">No officer data</td></tr>'}
            </tbody>
        </table>
    `;
}

function printReport() {
    window.print();
}

async function exportToPDF() {
    try {
        const startDate = document.getElementById('reportStartDate').value;
        const endDate = document.getElementById('reportEndDate').value;
        
        if (!startDate || !endDate) {
            alert('❌ Please select both start and end dates for the report');
            return;
        }
        
        // Show loading message
        const exportBtn = document.getElementById('exportPDFBtn');
        const originalText = exportBtn.textContent;
        exportBtn.textContent = '⏳ Generating PDF...';
        exportBtn.disabled = true;
        
        // Download PDF from backend
        const token = localStorage.getItem('token');
        const response = await fetch(`${API_BASE_URL}/accountant/report/pdf?start_date=${startDate}&end_date=${endDate}`, {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (!response.ok) {
            throw new Error('Failed to generate PDF');
        }
        
        // Create blob from response
        const blob = await response.blob();
        
        // Create download link
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `Financial_Report_${startDate}_to_${endDate}.pdf`;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
        
        // Reset button
        exportBtn.textContent = originalText;
        exportBtn.disabled = false;
        
        alert('✅ PDF report downloaded successfully!');
    } catch (error) {
        console.error('Error exporting PDF:', error);
        alert('❌ Failed to generate PDF. Please try again.');
        
        // Reset button
        const exportBtn = document.getElementById('exportPDFBtn');
        exportBtn.textContent = '📄 Export to PDF';
        exportBtn.disabled = false;
    }
}

// ============================================
// MODAL FUNCTIONS
// ============================================

function showLoanTopupModal() {
    // Get current loan data and check application fee status
    const currentLoan = window.currentClientLoan || {};
    apiCall('/client/loan-overview').then(overview => {
        const feeStatus = overview.data?.application_fee_status || 'pending';
        if (feeStatus !== 'paid') {
            alert('You must pay the application fee before requesting a top-up loan. Please pay at the office or via mobile money, then try again.');
            return;
        }
        document.getElementById('currentLoanAmount').textContent = 
            `UGX ${Number(currentLoan.amount || 0).toLocaleString()}`;
        document.getElementById('currentAmountPaid').textContent = 
            `UGX ${Number(currentLoan.amount_paid || 0).toLocaleString()}`;
        document.getElementById('currentBalance').textContent = 
            `UGX ${Number(currentLoan.balance || 0).toLocaleString()}`;
        document.getElementById('newLoanAmount').textContent = 
            `UGX ${Number(currentLoan.balance || 0).toLocaleString()}`;
        openModal('loanTopupModal');
        // Attach submit handler for top-up form
        const topupForm = document.getElementById('loanTopupForm');
        if (topupForm && !topupForm.hasAttribute('data-handler-attached')) {
            topupForm.setAttribute('data-handler-attached', 'true');
            topupForm.addEventListener('submit', async function(e) {
                e.preventDefault();
                // Collect top-up amount and other required fields
                const topupAmount = parseFloat(document.getElementById('topupAmount').value) || 0;
                if (topupAmount <= 0) {
                    alert('Please enter a valid top-up amount.');
                    return;
                }
                // Call backend to request top-up
                try {
                    const response = await apiCall('/client/request-topup', 'POST', { amount: topupAmount });
                    if (response.success) {
                        alert('✅ Top-up request submitted! Please pay the application fee to proceed.');
                        closeModal('loanTopupModal');
                        // Reload loan overview to update button state
                        await loadLoanOverview();
                    } else {
                        alert('❌ ' + (response.message || 'Failed to submit top-up request'));
                    }
                } catch (err) {
                    alert('❌ Error submitting top-up request.');
                }
            });
        }
    });
}

// Create user form
const createUserForm = document.getElementById('createUserForm');
if (createUserForm) {
    createUserForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const submitButton = e.target.querySelector('button[type="submit"]');
        const originalText = submitButton.textContent;
        submitButton.disabled = true;
        submitButton.textContent = 'Creating...';
        
        const data = {
            name: document.getElementById('userName').value,
            email: document.getElementById('userEmail').value,
            phone: document.getElementById('userPhone').value,
            password: document.getElementById('userPassword').value,
            role: document.getElementById('userRole').value
        };
        
        try {
            const response = await apiCall('/manager/users', 'POST', data);
            if (response.success) {
                alert('✅ User created successfully!\n\nUser can now login with:\nEmail: ' + data.email + '\nPassword: ' + data.password);
                closeModal('createUserModal');
                createUserForm.reset();
                loadUsers();
            } else {
                alert('❌ Failed to create user: ' + (response.message || 'Unknown error'));
            }
        } catch (error) {
            console.error('Error creating user:', error);
            alert('❌ Failed to create user: ' + error.message);
        } finally {
            submitButton.disabled = false;
            submitButton.textContent = originalText;
        }
    });
}

// Approve loan form
const approveLoanForm = document.getElementById('approveLoanForm');
if (approveLoanForm) {
    approveLoanForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const loanId = document.getElementById('approveLoanId').value;
        const officerId = document.getElementById('assignOfficer').value;
        const disbursedAmount = document.getElementById('disbursedAmount').value;
        
        if (!officerId) {
            alert('⚠️ Please select a credit officer');
            return;
        }
        
        if (!disbursedAmount || disbursedAmount <= 0) {
            alert('⚠️ Please enter a valid disbursed amount');
            return;
        }
        
        const submitButton = e.target.querySelector('button[type="submit"]');
        const originalText = submitButton.textContent;
        submitButton.disabled = true;
        submitButton.textContent = 'Approving...';
        
        try {
            const response = await apiCall(`/manager/loans/${loanId}/status`, 'PUT', {
                status: 'approved',
                officer_id: officerId,
                disbursed_amount: disbursedAmount
            });
            
            if (response.success) {
                alert('✅ Loan approved successfully!');
                closeModal('approveLoanModal');
                approveLoanForm.reset();
                loadPendingLoans();
                loadManagerStats(); // Refresh dashboard stats
            } else {
                alert('❌ Failed to approve loan: ' + (response.message || 'Unknown error'));
            }
        } catch (error) {
            console.error('Error approving loan:', error);
            alert('❌ Failed to approve loan: ' + error.message);
        } finally {
            submitButton.disabled = false;
            submitButton.textContent = originalText;
        }
    });
}

// Loan topup form
    const loanTopupForm = document.getElementById('loanTopupForm');
    if (loanTopupForm) {
        loanTopupForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const data = {
                amount: document.getElementById('topupAmount').value,
                loan_type: document.getElementById('topupType').value,
                repayment_mode: document.getElementById('topupRepaymentMode').value
            };
            
            try {
                const response = await apiCall('/client/loan-topup', 'POST', data);
                if (response.success) {
                    alert('Loan top-up request submitted successfully!');
                    closeModal('loanTopupModal');
                    loanTopupForm.reset();
                    loadLoanOverview();
                }
            } catch (error) {
                alert('Failed to submit loan request');
            }
        });
    }
    
    // Client loan application form
    const clientLoanApplicationForm = document.getElementById('clientLoanApplicationForm');
    if (clientLoanApplicationForm) {
        clientLoanApplicationForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const messageDiv = document.getElementById('loanApplicationMessage');
            messageDiv.textContent = '';
            messageDiv.style.display = 'none';
            
            const data = {
                amount: document.getElementById('loanAmount').value,
                loan_type: document.getElementById('loanType').value,
                purpose: document.getElementById('loanPurpose').value,
                duration_months: document.getElementById('loanDuration').value,
                repayment_mode: document.getElementById('repaymentMode').value,
                collateral_type: document.getElementById('collateralType').value || null,
                collateral_value: document.getElementById('collateralValue').value || null,
                collateral_description: document.getElementById('collateralDescription').value || null
            };
            
            try {
                const response = await apiCall('/client/apply-loan', 'POST', data);
                if (response.success) {
                    messageDiv.className = 'message success';
                    messageDiv.textContent = '✅ Loan application submitted successfully! Your credit officer will review it soon.';
                    messageDiv.style.display = 'block';
                    
                    // Reset form and preview
                    clientLoanApplicationForm.reset();
                    document.getElementById('loanPreview').style.display = 'none';
                    const inputs = clientLoanApplicationForm.querySelectorAll('input, select, textarea');
                    inputs.forEach(input => input.disabled = false);
                    document.getElementById('submitLoanBtn').textContent = '📤 Submit Loan Application';
                    
                    // Reload the section to show the pending application
                    setTimeout(() => {
                        loadClientLoanApplicationStatus();
                        messageDiv.style.display = 'none';
                    }, 3000);
                } else {
                    messageDiv.className = 'message error';
                    messageDiv.textContent = '❌ ' + (response.message || 'Failed to submit application');
                    messageDiv.style.display = 'block';
                    
                    // Re-enable form for editing
                    editLoanApplication();
                }
            } catch (error) {
                console.error('Loan application error:', error);
                messageDiv.className = 'message error';
                messageDiv.textContent = '❌ Server error. Please try again. Error: ' + (error.message || 'Unknown error');
                messageDiv.style.display = 'block';
                
                // Re-enable form for editing
                editLoanApplication();
            }
        });
    }
    
    // Add transaction form
    const addTransactionForm = document.getElementById('addTransactionForm');
    if (addTransactionForm) {
        addTransactionForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const data = {
                transaction_type: document.getElementById('transactionType').value,
                amount: document.getElementById('transactionAmount').value,
                description: document.getElementById('transactionDescription').value
            };
            
            try {
                const response = await apiCall('/accountant/transactions', 'POST', data);
                if (response.success) {
                    alert('Transaction added successfully!');
                    closeModal('addTransactionModal');
                    addTransactionForm.reset();
                    loadTransactions();
                    loadFinancialSummary();
                }
            } catch (error) {
                alert('Failed to add transaction');
            }
        });
    }
    
    // Add petty cash form (expense)
    const addPettyCashForm = document.getElementById('addPettyCashForm');
    if (addPettyCashForm) {
        addPettyCashForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const data = {
                amount: document.getElementById('pettyCashAmount').value,
                description: document.getElementById('pettyCashDescription').value
            };
            
            try {
                // Determine endpoint based on user role
                const userRole = localStorage.getItem('userRole');
                const endpoint = userRole === 'manager' ? '/manager/petty-cash' : '/accountant/petty-cash';
                
                const response = await apiCall(endpoint, 'POST', data);
                if (response.success) {
                    alert('✅ Petty cash expense recorded successfully!');
                    closeModal('addPettyCashModal');
                    addPettyCashForm.reset();
                    
                    // Reload appropriate list
                    if (userRole === 'manager') {
                        loadManagerPettyCash();
                    } else {
                        loadPettyCash();
                        loadFinancialSummary();
                    }
                }
            } catch (error) {
                alert('❌ Failed to record petty cash expense. ' + (error.message || ''));
            }
        });
    }
    
    // Add funds form
    const addFundsForm = document.getElementById('addFundsForm');
    if (addFundsForm) {
        addFundsForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const data = {
                account_code: document.getElementById('fundsAccount').value,
                amount: parseFloat(document.getElementById('fundsAmount').value),
                source: document.getElementById('fundsSource').value,
                description: document.getElementById('fundsDescription').value
            };
            
            try {
                const response = await apiCall('/accountant/add-funds', 'POST', data);
                if (response.success) {
                    alert('✅ Funds added successfully!');
                    closeModal('addFundsModal');
                    addFundsForm.reset();
                    loadFinancialSummary();
                    loadTransactions();
                    loadAccountBalances(); // Refresh account balances cards
                }
            } catch (error) {
                alert('❌ Failed to add funds. ' + (error.message || ''));
            }
        });
    }
    
    // Allocate petty cash form
    const allocatePettyCashForm = document.getElementById('allocatePettyCashForm');
    if (allocatePettyCashForm) {
        allocatePettyCashForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const data = {
                amount: document.getElementById('allocateAmount').value,
                description: document.getElementById('allocateDescription').value
            };
            
            try {
                const response = await apiCall('/accountant/petty-cash/allocate', 'POST', data);
                if (response.success) {
                    alert(`✅ ${response.message}`);
                    closeModal('allocatePettyCashModal');
                    allocatePettyCashForm.reset();
                    loadPettyCash();
                    loadFinancialSummary();
                }
            } catch (error) {
                alert('❌ Failed to allocate petty cash. ' + (error.message || ''));
            }
        });
    }
    
    // Allocate account funds form
    const allocateAccountForm = document.getElementById('allocateAccountForm');
    if (allocateAccountForm) {
        allocateAccountForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const data = {
                account_code: document.getElementById('accountCodeInput').value,
                amount: document.getElementById('accountAllocateAmount').value,
                description: document.getElementById('accountAllocateDescription').value
            };
            
            try {
                const response = await apiCall('/accountant/account-allocations/allocate', 'POST', data);
                if (response.success) {
                    alert(`✅ ${response.message}`);
                    closeModal('allocateAccountModal');
                    allocateAccountForm.reset();
                    loadAccountBalances();
                    loadFinancialSummary();
                }
            } catch (error) {
                alert('❌ Failed to allocate account funds. ' + (error.message || ''));
            }
        });
    }
    
    // Loan fees configuration form (Manager) - Only Application Fee now
    const loanFeesForm = document.getElementById('loanFeesForm');
    if (loanFeesForm) {
        loanFeesForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const applicationFee = document.getElementById('applicationFeeAmount').value;
            
            try {
                // Update application fee only (processing fee is now automatic)
                await apiCall('/manager/system-settings', 'PUT', {
                    setting_key: 'default_application_fee',
                    setting_value: applicationFee
                });
                
                const messageDiv = document.getElementById('loanFeesMessage');
                messageDiv.textContent = '✅ Application fee updated successfully!';
                messageDiv.className = 'message success';
                messageDiv.style.display = 'block';
                
                setTimeout(() => {
                    messageDiv.style.display = 'none';
                }, 3000);
            } catch (error) {
                const messageDiv = document.getElementById('loanFeesMessage');
                messageDiv.textContent = '❌ Failed to update application fee. ' + (error.message || '');
                messageDiv.className = 'message error';
                messageDiv.style.display = 'block';
            }
        });
    }
    
    // Assign officer form
    const assignOfficerForm = document.getElementById('assignOfficerForm');
    if (assignOfficerForm) {
        assignOfficerForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const clientId = document.getElementById('reassignClientId').value;
            const officerId = document.getElementById('selectOfficer').value;
            
            if (!officerId) {
                alert('⚠️ Please select an officer');
                return;
            }
            
            try {
                const response = await apiCall('/enrollment/assign', 'POST', {
                    clientId: parseInt(clientId),
                    officerId: parseInt(officerId)
                });
                
                if (response.success) {
                    alert('✅ Client reassigned successfully!');
                    closeModal('assignOfficerModal');
                    assignOfficerForm.reset();
                    loadAllClients();
                }
            } catch (error) {
                alert('❌ Failed to reassign client');
            }
        });
    }
    
    // Edit client form
    const editClientForm = document.getElementById('editClientForm');
    if (editClientForm) {
        editClientForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const clientId = document.getElementById('editClientId').value;
            const messageDiv = document.getElementById('editClientMessage');
            
            const data = {
                name: document.getElementById('editClientName').value,
                email: document.getElementById('editClientEmail').value,
                phone: document.getElementById('editClientPhone').value,
                national_id: document.getElementById('editClientNationalId').value,
                village: document.getElementById('editClientVillage').value,
                parish: document.getElementById('editClientParish').value,
                division: document.getElementById('editClientDivision').value,
                occupation: document.getElementById('editClientOccupation').value,
                status: document.getElementById('editClientStatus').value
            };
            
            try {
                const response = await apiCall(`/enrollment/clients/${clientId}`, 'PUT', data);
                
                if (response.success) {
                    messageDiv.className = 'message success';
                    messageDiv.textContent = '✅ Client updated successfully!';
                    
                    setTimeout(() => {
                        closeModal('editClientModal');
                        messageDiv.textContent = '';
                        loadAllClients();
                    }, 1500);
                } else {
                    messageDiv.className = 'message error';
                    messageDiv.textContent = '❌ ' + (response.error || 'Failed to update client');
                }
            } catch (error) {
                messageDiv.className = 'message error';
                messageDiv.textContent = '❌ Error: ' + error.message;
            }
        });
    }
    
    // Record payment form - DISABLED: Only accountant can record payments
    // const recordPaymentForm = document.getElementById('recordPaymentForm');
    // if (recordPaymentForm) {
    //     recordPaymentForm.addEventListener('submit', async (e) => {
    //         e.preventDefault();
    //         
    //         const data = {
    //             loanId: document.getElementById('paymentLoanId').value,
    //             amount: document.getElementById('paymentAmount').value,
    //             payment_method: document.getElementById('paymentMethod').value,
    //             notes: document.getElementById('paymentNotes').value
    //         };
    //         
    //         try {
    //             const response = await apiCall('/officer/payments', 'POST', data);
    //             if (response.success) {
    //                 alert('Payment recorded successfully!');
    //                 closeModal('recordPaymentModal');
    //                 recordPaymentForm.reset();
    //                 loadAssignedClients();
    //             }
    //         } catch (error) {
    //             alert('Failed to record payment');
    //         }
    //     });
    // }

// Modal functions
function openModal(modalId) {
    document.getElementById(modalId).classList.add('active');
}

function closeModal(modalId) {
    document.getElementById(modalId).classList.remove('active');
}

// Download client details as PDF
function downloadClientDetailsPDF() {
    // Get the client details modal content
    const content = document.getElementById('clientDetailsContent');
    
    if (!content) {
        alert('No client details to download');
        return;
    }
    
    // Create a printable version
    const printWindow = window.open('', '_blank');
    printWindow.document.write(`
        <!DOCTYPE html>
        <html>
        <head>
            <title>Client Details</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    padding: 20px;
                    max-width: 800px;
                    margin: 0 auto;
                }
                h3 {
                    color: #2563eb;
                    border-bottom: 2px solid #2563eb;
                    padding-bottom: 10px;
                }
                p {
                    margin: 8px 0;
                    line-height: 1.6;
                }
                strong {
                    font-weight: bold;
                    color: #333;
                }
                .section {
                    margin: 20px 0;
                    padding: 15px;
                    border: 1px solid #ddd;
                    border-radius: 8px;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin: 10px 0;
                }
                table th, table td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: left;
                }
                table th {
                    background-color: #f0f0f0;
                }
                @media print {
                    button {
                        display: none;
                    }
                }
            </style>
        </head>
        <body>
            <h2>Starting Point Financial Institution</h2>
            <h3>Client Details Report</h3>
            <p><strong>Generated:</strong> ${new Date().toLocaleString()}</p>
            <hr>
            ${content.innerHTML}
            <br>
            <button onclick="window.print()" style="padding: 10px 20px; background: #2563eb; color: white; border: none; border-radius: 5px; cursor: pointer;">Print PDF</button>
            <button onclick="window.close()" style="padding: 10px 20px; background: #666; color: white; border: none; border-radius: 5px; cursor: pointer; margin-left: 10px;">Close</button>
        </body>
        </html>
    `);
    printWindow.document.close();
}

// Close modals when clicking outside
window.onclick = function(event) {
    if (event.target.classList.contains('modal')) {
        event.target.classList.remove('active');
    }
}

// ==================== CLIENT ENROLLMENT ====================

// Load officers for dropdown
async function loadOfficersDropdown() {
    try {
        const response = await apiCall('/manager/officers', 'GET');
        
        const dropdown = document.getElementById('enrollOfficer');
        if (dropdown && response.success) {
            dropdown.innerHTML = '<option value="">Select Officer...</option>';
            
            // Add all officers to dropdown
            if (response.officers && response.officers.length > 0) {
                response.officers.forEach(officer => {
                    const option = document.createElement('option');
                    option.value = officer.id;
                    option.textContent = officer.name;
                    dropdown.appendChild(option);
                });
            } else {
                dropdown.innerHTML += '<option value="">No officers available</option>';
            }
        }
    } catch (error) {
        console.error('Error loading officers:', error);
        const dropdown = document.getElementById('enrollOfficer');
        if (dropdown) {
            dropdown.innerHTML = '<option value="">Error loading officers</option>';
        }
    }
}

// Handle enrollment form submission (Manager)
const enrollForm = document.getElementById('enrollClientForm');
if (enrollForm) {
    enrollForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const formData = new FormData(enrollForm);
        const messageDiv = document.getElementById('enrollMessage');
        
        try {
            const response = await fetch('/api/enrollment/enroll', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${authToken}`
                },
                body: formData
            });
            
            const data = await response.json();
            
            if (response.ok && data.success) {
                messageDiv.className = 'message success';
                messageDiv.textContent = data.message || '✅ Client enrolled successfully!';
                enrollForm.reset();
                setTimeout(() => {
                    messageDiv.textContent = '';
                    showManagerSection('clients');
                }, 2000);
            } else {
                messageDiv.className = 'message error';
                messageDiv.textContent = '❌ ' + (data.error || data.message || 'Enrollment failed');
            }
        } catch (error) {
            messageDiv.className = 'message error';
            messageDiv.textContent = '❌ Error: ' + error.message;
        }
    });
}

// Handle enrollment form submission (Officer)
const enrollFormOfficer = document.getElementById('enrollClientFormOfficer');
if (enrollFormOfficer) {
    enrollFormOfficer.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const formData = new FormData(enrollFormOfficer);
        const messageDiv = document.getElementById('enrollMessageOfficer');
        
        try {
            const response = await fetch('/api/enrollment/enroll', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${authToken}`
                },
                body: formData
            });
            
            const data = await response.json();
            
            if (response.ok && data.success) {
                messageDiv.className = 'message success';
                messageDiv.textContent = data.message || '✅ Client enrolled successfully!';
                enrollFormOfficer.reset();
                setTimeout(() => {
                    messageDiv.textContent = '';
                    showOfficerSection('clients');
                }, 2000);
            } else {
                messageDiv.className = 'message error';
                messageDiv.textContent = '❌ ' + (data.error || data.message || 'Enrollment failed');
            }
        } catch (error) {
            messageDiv.className = 'message error';
            messageDiv.textContent = '❌ Error: ' + error.message;
        }
    });
}

// Load all clients (Manager)
async function loadAllClients() {
    try {
        const response = await apiCall('/enrollment/clients', 'GET');
        
        if (response.success) {
            const table = document.getElementById('clientsTable');
            if (!table) return;
            
            if (response.clients.length === 0) {
                table.innerHTML = '<tr><td colspan="6" class="text-center">No clients found</td></tr>';
                return;
            }
            
            table.innerHTML = response.clients.map(client => `
                <tr>
                    <td>${client.name}</td>
                    <td>${client.phone || 'N/A'}</td>
                    <td>${client.village || 'N/A'}</td>
                    <td>${client.officer_name || 'Unassigned'}</td>
                    <td><span class="badge ${client.status === 'active' ? 'badge-success' : 'badge-secondary'}">${client.status}</span></td>
                    <td>
                        <button onclick="viewClientDetails(${client.id})" class="btn btn-sm">View</button>
                        <button onclick="showEditClientModal(${client.id})" class="btn btn-sm" style="background: #ffc107; color: #000;">Edit</button>
                        <button onclick="showAssignOfficerModal(${client.id})" class="btn btn-sm">Reassign</button>
                        <button onclick="showResetPasswordModal(${client.id})" class="btn btn-sm">Reset Pwd</button>
                    </td>
                </tr>
            `).join('');
        }
    } catch (error) {
        console.error('Error loading clients:', error);
    }
}

// Show assign officer modal
async function showAssignOfficerModal(clientId) {
    document.getElementById('reassignClientId').value = clientId;
    
    // Load officers list
    try {
        const response = await apiCall('/manager/officers', 'GET');
        const select = document.getElementById('selectOfficer');
        select.innerHTML = '<option value="">-- Select Officer --</option>';
        
        if (response.success && response.officers) {
            response.officers.forEach(officer => {
                const option = document.createElement('option');
                option.value = officer.id;
                option.textContent = `${officer.name} (${officer.email})`;
                select.appendChild(option);
            });
        }
    } catch (error) {
        console.error('Error loading officers:', error);
    }
    
    openModal('assignOfficerModal');
}

// View client loan details (for officer dashboard)
async function viewClientLoanDetails(clientId, loanId) {
    try {
        console.log('Loading loan details for client:', clientId, 'loan:', loanId);
        
        // Get loan details
        const loanResponse = await apiCall(`/officer/loan-details/${loanId}`);
        
        if (loanResponse.success) {
            const loan = loanResponse.data;
            
            // Get client details
            const clientResponse = await apiCall(`/officer/clients/${clientId}`);
            const client = clientResponse.success ? clientResponse.data.client : null;
            const guarantors = clientResponse.success ? clientResponse.data.guarantors : [];
            const repaymentHistory = clientResponse.success ? clientResponse.data.repaymentHistory : [];
            
            const guarantorsHtml = guarantors && guarantors.length > 0 
                ? guarantors.map((g, index) => `
                    <div style="background: ${index === 0 ? '#f0f9ff' : '#f0fdf4'}; padding: 15px; border-radius: 8px; margin-bottom: 10px;">
                        <h4 style="margin: 0 0 10px 0;">Guarantor ${index + 1} - ${g.name}</h4>
                        <p><strong>Phone:</strong> <a href="tel:${g.phone}" style="color: #2563eb;">${g.phone}</a></p>
                        ${g.email ? `<p><strong>Email:</strong> ${g.email}</p>` : ''}
                        <p><strong>Relationship:</strong> ${g.relationship}</p>
                        ${g.address ? `<p><strong>Address:</strong> ${g.address}</p>` : ''}
                        ${g.national_id ? `<p><strong>National ID:</strong> ${g.national_id}</p>` : ''}
                    </div>
                `).join('')
                : '<p style="color: #666;">No guarantors on record</p>';
            
            const content = document.getElementById('clientDetailsContent');
            content.innerHTML = `
                <div class="loan-details">
                    <h3>💰 Loan Details</h3>
                    <div style="background: #f9fafb; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
                        <p><strong>Loan Type:</strong> ${loan.loan_type}</p>
                        <p><strong>Amount:</strong> <span style="color: #2563eb; font-size: 1.2em; font-weight: bold;">UGX ${Number(loan.amount).toLocaleString()}</span></p>
                        <p><strong>Balance:</strong> <span style="color: ${loan.balance > 0 ? '#dc3545' : '#28a745'}; font-size: 1.2em; font-weight: bold;">UGX ${Number(loan.balance || 0).toLocaleString()}</span></p>
                        <p><strong>Amount Paid:</strong> UGX ${Number(loan.amount_paid || 0).toLocaleString()}</p>
                        <p><strong>Status:</strong> <span class="badge badge-${loan.status}">${loan.status}</span></p>
                        <p><strong>Duration:</strong> ${loan.duration_months} months</p>
                        <p><strong>Repayment Mode:</strong> ${loan.repayment_mode}</p>
                        ${loan.next_payment_date ? `<p><strong>Next Payment Date:</strong> ${new Date(loan.next_payment_date).toLocaleDateString()}</p>` : ''}
                        ${loan.purpose ? `<p><strong>Purpose:</strong> ${loan.purpose}</p>` : ''}
                        <p><strong>Application Date:</strong> ${new Date(loan.created_at).toLocaleDateString()}</p>
                    </div>
                </div>
                
                ${client ? `
                <div class="client-info">
                    <h3>👤 Client Information</h3>
                    <div style="background: #f0f9ff; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
                        <p><strong>Name:</strong> ${client.name}</p>
                        <p><strong>Phone:</strong> <a href="tel:${client.phone}" style="color: #2563eb;">${client.phone}</a></p>
                        <p><strong>Email:</strong> ${client.email}</p>
                        ${client.national_id ? `<p><strong>National ID:</strong> ${client.national_id}</p>` : ''}
                        ${client.village ? `<p><strong>Village:</strong> ${client.village}</p>` : ''}
                        ${client.occupation ? `<p><strong>Occupation:</strong> ${client.occupation}</p>` : ''}
                    </div>
                </div>
                ` : ''}
                
                <div class="guarantors-section" style="margin-top: 20px;">
                    <h3>👥 Guarantors</h3>
                    ${guarantorsHtml}
                </div>
                
                <div class="repayment-history" style="margin-top: 20px;">
                    <h3>📝 Repayment History</h3>
                    ${repaymentHistory.length > 0 ? `
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Amount</th>
                                <th>Method</th>
                                <th>Notes</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${repaymentHistory.map(payment => `
                                <tr>
                                    <td>${new Date(payment.payment_date).toLocaleDateString()}</td>
                                    <td>UGX ${Number(payment.amount).toLocaleString()}</td>
                                    <td>${payment.payment_method}</td>
                                    <td>${payment.notes || '-'}</td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                    ` : '<p>No repayments yet</p>'}
                </div>
                
                <div style="margin-top: 20px; text-align: center;">
                    <button class="btn" onclick="closeModal('clientDetailsModal')">Close</button>
                </div>
            `;
            
            openModal('clientDetailsModal');
        } else {
            alert('❌ Failed to load loan details');
        }
    } catch (error) {
        console.error('Error loading loan details:', error);
        alert('❌ Failed to load loan details');
    }
}

// View client details
async function viewClientDetails(clientId) {
    try {
        console.log('Loading client details for ID:', clientId);
        
        // Determine which endpoint to use based on user role
        let response;
        const userRole = localStorage.getItem('userRole');
        
        if (userRole === 'officer') {
            console.log('Using officer endpoint');
            response = await apiCall(`/officer/clients/${clientId}`);
        } else {
            console.log('Using enrollment endpoint for manager');
            response = await apiCall(`/enrollment/clients/${clientId}`);
        }
        
        console.log('Response received:', response);
        
        if (response.success) {
            const content = document.getElementById('clientDetailsContent');
            
            // Handle both response formats
            const client = response.data ? response.data.client : response.client;
            const guarantors = response.data ? response.data.guarantors : response.guarantors || [];
            const loans = response.data ? response.data.loans : [];
            const repaymentHistory = response.data ? response.data.repaymentHistory : [];
            
            const guarantorsHtml = guarantors && guarantors.length > 0 
                ? guarantors.map((g, index) => `
                    <div style="background: ${index === 0 ? '#f0f9ff' : '#f0fdf4'}; padding: 15px; border-radius: 8px; margin-bottom: 10px;">
                        <h4 style="margin: 0 0 10px 0;">Guarantor ${index + 1} - ${g.name}</h4>
                        <p><strong>Phone:</strong> <a href="tel:${g.phone}" style="color: #2563eb;">${g.phone}</a></p>
                        ${g.email ? `<p><strong>Email:</strong> ${g.email}</p>` : ''}
                        <p><strong>Relationship:</strong> ${g.relationship}</p>
                        ${g.address ? `<p><strong>Address:</strong> ${g.address}</p>` : ''}
                        ${g.national_id ? `<p><strong>National ID:</strong> ${g.national_id}</p>` : ''}
                    </div>
                `).join('')
                : '<p style="color: #666;">No guarantors on record</p>';
            
            content.innerHTML = `
                <div class="client-info">
                    <h3>📋 Client Information</h3>
                    <p><strong>Name:</strong> ${client.name}</p>
                    <p><strong>Email:</strong> ${client.email}</p>
                    <p><strong>Phone:</strong> ${client.phone || 'N/A'}</p>
                    <p><strong>National ID:</strong> ${client.national_id || 'N/A'}</p>
                    ${client.village ? `<p><strong>Village:</strong> ${client.village}</p>` : ''}
                    ${client.parish ? `<p><strong>Parish:</strong> ${client.parish}</p>` : ''}
                    ${client.division ? `<p><strong>Division:</strong> ${client.division}</p>` : ''}
                    ${client.occupation ? `<p><strong>Occupation:</strong> ${client.occupation}</p>` : ''}
                    ${client.officer_name ? `<p><strong>Assigned Officer:</strong> ${client.officer_name}</p>` : ''}
                    ${client.status ? `<p><strong>Status:</strong> <span class="badge ${client.status === 'active' ? 'badge-success' : 'badge-secondary'}">${client.status}</span></p>` : ''}
                </div>
                
                <div class="guarantors-section" style="margin-top: 20px;">
                    <h3>👥 Guarantors (Alternative Contacts)</h3>
                    ${guarantorsHtml}
                </div>
                
                <div class="loan-history" style="margin-top: 20px;">
                    <h3>💰 Loan History</h3>
                    ${loans.length > 0 ? loans.map(loan => `
                        <div class="loan-item">
                            <p><strong>Type:</strong> ${loan.loan_type}</p>
                            <p><strong>Amount:</strong> UGX ${Number(loan.amount).toLocaleString()}</p>
                            <p><strong>Balance:</strong> UGX ${Number(loan.balance || 0).toLocaleString()}</p>
                            <p><strong>Status:</strong> <span class="badge">${loan.status}</span></p>
                        </div>
                    `).join('') : '<p>No loans</p>'}
                    <div id="clientLoanInfo"></div>
                </div>
                
                <div class="repayment-history" style="margin-top: 20px;">
                    <h3>📝 Recent Repayments</h3>
                    ${repaymentHistory.length > 0 ? `
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Amount</th>
                                <th>Method</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${repaymentHistory.map(payment => `
                                <tr>
                                    <td>${new Date(payment.payment_date).toLocaleDateString()}</td>
                                    <td>UGX ${Number(payment.amount).toLocaleString()}</td>
                                    <td>${payment.payment_method}</td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                    ` : '<p>No repayments yet</p>'}
                </div>
            `;
            
            openModal('clientDetailsModal');
            
            // Load additional loan info for managers
            if (!response.data) {
                loadClientLoans(clientId);
            }
        }
    } catch (error) {
        console.error('Error loading client details:', error);
        alert('❌ Failed to load client details');
    }
}

// Load client loans
async function loadClientLoans(clientId) {
    try {
        const response = await apiCall(`/enrollment/clients/${clientId}/loans`);
        const container = document.getElementById('clientLoanInfo');
        
        if (response.success && response.loans && response.loans.length > 0) {
            container.innerHTML = `
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Loan Type</th>
                            <th>Requested</th>
                            <th>Disbursed</th>
                            <th>Paid</th>
                            <th>Balance</th>
                            <th>Status</th>
                            <th>Date</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${response.loans.map(loan => `
                            <tr>
                                <td>${loan.loan_type}</td>
                                <td>UGX ${Number(loan.amount).toLocaleString()}</td>
                                <td>UGX ${Number(loan.disbursed_amount || loan.amount).toLocaleString()}</td>
                                <td>UGX ${Number(loan.amount_paid || 0).toLocaleString()}</td>
                                <td>UGX ${Number(loan.balance).toLocaleString()}</td>
                                <td><span class="badge badge-${loan.status}">${loan.status}</span></td>
                                <td>${new Date(loan.created_at).toLocaleDateString()}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            `;
        } else {
            container.innerHTML = '<p class="text-center">No loans found for this client</p>';
        }
    } catch (error) {
        console.error('Error loading client loans:', error);
        if (document.getElementById('clientLoanInfo')) {
            document.getElementById('clientLoanInfo').innerHTML = '<p class="text-center">Failed to load loans</p>';
        }
    }
}

// Show edit client modal
async function showEditClientModal(clientId) {
    try {
        const response = await apiCall(`/enrollment/clients/${clientId}`);
        
        if (response.success && response.client) {
            const client = response.client;
            
            // Populate form fields
            document.getElementById('editClientId').value = client.id;
            document.getElementById('editClientName').value = client.name || '';
            document.getElementById('editClientEmail').value = client.email || '';
            document.getElementById('editClientPhone').value = client.phone || '';
            document.getElementById('editClientNationalId').value = client.national_id || '';
            document.getElementById('editClientVillage').value = client.village || '';
            document.getElementById('editClientParish').value = client.parish || '';
            document.getElementById('editClientDivision').value = client.division || '';
            document.getElementById('editClientOccupation').value = client.occupation || '';
            document.getElementById('editClientStatus').value = client.status || 'active';
            
            openModal('editClientModal');
        } else {
            alert('❌ Failed to load client details');
        }
    } catch (error) {
        console.error('Error loading client for edit:', error);
        alert('❌ Failed to load client details');
    }
}

// Assign client to officer
async function assignClientToOfficer(clientId, officerId) {
    try {
        const response = await apiCall('/enrollment/assign', 'POST', {
            clientId,
            officerId
        });
        
        if (response.success) {
            alert('✅ Client reassigned successfully!');
            loadAllClients();
        }
    } catch (error) {
        alert('❌ Failed to reassign client');
    }
}

// Show reset password modal
function showResetPasswordModal(userId) {
    const newPassword = prompt('Enter new password for user:');
    if (newPassword) {
        resetEnrollmentPassword(userId, newPassword);
    }
}

// Reset user password
async function resetEnrollmentPassword(userId, newPassword) {
    try {
        const response = await apiCall('/enrollment/reset-password', 'POST', {
            userId,
            newPassword
        });
        
        if (response.success) {
            alert('✅ Password reset successfully!');
        }
    } catch (error) {
        alert('❌ Failed to reset password');
    }
}

// Setup password change form for manager
function setupPasswordChangeForm() {
    const form = document.getElementById('changePasswordForm');
    form.onsubmit = async (e) => {
        e.preventDefault();
        
        const currentPassword = document.getElementById('currentPassword').value;
        const newPassword = document.getElementById('newPassword').value;
        const confirmPassword = document.getElementById('confirmPassword').value;
        const messageDiv = document.getElementById('passwordChangeMessage');
        
        // Validate passwords match
        if (newPassword !== confirmPassword) {
            messageDiv.textContent = '❌ New passwords do not match!';
            messageDiv.className = 'message error';
            messageDiv.style.display = 'block';
            return;
        }
        
        // Validate password length
        if (newPassword.length < 6) {
            messageDiv.textContent = '❌ Password must be at least 6 characters!';
            messageDiv.className = 'message error';
            messageDiv.style.display = 'block';
            return;
        }
        
        try {
            const response = await apiCall('/manager/change-password', 'PUT', {
                currentPassword,
                newPassword
            });
            
            if (response.success) {
                messageDiv.textContent = '✅ Password changed successfully!';
                messageDiv.className = 'message success';
                messageDiv.style.display = 'block';
                form.reset();
            } else {
                messageDiv.textContent = '❌ ' + (response.message || 'Failed to change password');
                messageDiv.className = 'message error';
                messageDiv.style.display = 'block';
            }
        } catch (error) {
            messageDiv.textContent = '❌ Failed to change password. ' + (error.message || 'Please try again.');
            messageDiv.className = 'message error';
            messageDiv.style.display = 'block';
        }
    };
}

// Load account balances for accountant dashboard
async function loadAccountBalances() {
    try {
        const response = await apiCall('/accountant/account-balances');
        if (response.success) {
            const balances = response.data;
            
            balances.forEach(account => {
                const balance = Number(account.balance).toLocaleString();
                switch(account.account_code) {
                    case '1110':
                        document.getElementById('cashOnHandBalance').textContent = `UGX ${balance}`;
                        break;
                    case '1120':
                        document.getElementById('bankAccountBalance').textContent = `UGX ${balance}`;
                        break;
                    case '1130':
                        document.getElementById('mobileMoneyBalance').textContent = `UGX ${balance}`;
                        break;
                    case '1140':
                        document.getElementById('pettyCashBalanceCard').textContent = `UGX ${balance}`;
                        break;
                    case '1210':
                        document.getElementById('loanReceivablesBalance').textContent = `UGX ${balance}`;
                        break;
                }
            });
        }
        
        // Removed loadAccountAllocations() - simplified for easier use
    } catch (error) {
        console.error('Error loading account balances:', error);
    }
}

// Load account allocations (Cash, Bank, Mobile Money)
async function loadAccountAllocations() {
    try {
        const response = await apiCall('/accountant/account-allocations');
        if (response.success) {
            const container = document.getElementById('accountAllocationsContainer');
            container.innerHTML = '';
            
            response.data.forEach(account => {
                const allocated = Number(account.allocated_amount);
                const current = Number(account.current_balance);
                const spent = Number(account.spent);
                const percentage = allocated > 0 ? ((current / allocated) * 100).toFixed(1) : 0;
                
                // Map account codes to emojis
                let icon = '💰'; // Default
                if (account.account_code === '1110' || account.account_code === '1101' || account.account_code === '1102') {
                    icon = '💵'; // Cash
                } else if (account.account_code === '1120') {
                    icon = '🏦'; // Bank
                } else if (account.account_code === '1130') {
                    icon = '📱'; // Mobile Money
                } else if (account.account_code === '1210' || account.account_code === '1220') {
                    icon = '📊'; // Receivables
                } else if (account.account_code === '1140') {
                    icon = '💰'; // Petty Cash
                }
                
                const accountCard = document.createElement('div');
                accountCard.className = 'account-allocation-card';
                accountCard.style.cssText = 'border: 1px solid #ddd; border-radius: 8px; padding: 20px; margin-bottom: 20px; background: white;';
                
                accountCard.innerHTML = `
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                        <h4>${icon} ${account.account_name}</h4>
                        <div style="display: flex; gap: 10px;">
                            <button onclick="showAllocateAccountModal('${account.account_code}', '${account.account_name}')" class="btn btn-sm btn-primary">💵 Allocate</button>
                            <button onclick="topUpAccountFunds('${account.account_code}', '${account.account_name}')" class="btn btn-sm btn-success">➕ Top Up</button>
                        </div>
                    </div>
                    
                    <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 15px; margin-bottom: 15px;">
                        <div>
                            <p style="margin: 0; color: #666; font-size: 14px;">Allocated</p>
                            <p style="margin: 5px 0 0 0; font-size: 18px; font-weight: bold;">UGX ${allocated.toLocaleString()}</p>
                        </div>
                        <div>
                            <p style="margin: 0; color: #666; font-size: 14px;">Current Balance</p>
                            <p style="margin: 5px 0 0 0; font-size: 18px; font-weight: bold; color: ${current > 0 ? '#4CAF50' : '#f44336'};">UGX ${current.toLocaleString()}</p>
                        </div>
                        <div>
                            <p style="margin: 0; color: #666; font-size: 14px;">Used</p>
                            <p style="margin: 5px 0 0 0; font-size: 18px; font-weight: bold; color: #ff9800;">UGX ${spent.toLocaleString()}</p>
                        </div>
                    </div>
                    
                    <div style="background: #f5f5f5; height: 20px; border-radius: 10px; overflow: hidden;">
                        <div style="background: linear-gradient(90deg, #4CAF50, #8BC34A); height: 100%; width: ${percentage}%; transition: width 0.3s;"></div>
                    </div>
                    <p style="margin: 5px 0 0 0; font-size: 12px; color: #666; text-align: right;">${percentage}% remaining</p>
                `;
                
                container.appendChild(accountCard);
            });
        }
    } catch (error) {
        console.error('Error loading account allocations:', error);
    }
}

// Load approved loans ready for disbursement
async function loadApprovedLoans() {
    await loadApprovedLoansForDisbursement();
}

async function loadApprovedLoansForDisbursement() {
    try {
        const response = await apiCall('/accountant/approved-loans');
        if (response.success) {
            console.log('Approved loans data:', response.data); // Debug log
            const tbody = document.getElementById('approvedLoansTable');
            tbody.innerHTML = '';
            
            if (response.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="9" class="text-center">No loans ready for disbursement</td></tr>';
                return;
            }
            
            response.data.forEach(loan => {
                console.log('Processing loan:', loan); // Debug each loan
                const row = document.createElement('tr');
                const requestedAmount = Number(loan.requested_amount || loan.amount).toLocaleString();
                const approvedAmount = Number(loan.disbursed_amount || loan.amount).toLocaleString();
                const approvedDate = new Date(loan.approved_at).toLocaleDateString();
                
                // Check application fee status
                const feeStatus = loan.application_fee_status || 'pending';
                console.log(`Loan ${loan.loan_id}: fee_status = ${feeStatus}`); // Debug fee status
                const canDisburse = (feeStatus === 'paid' || feeStatus === 'waived');
                console.log(`Loan ${loan.loan_id}: canDisburse = ${canDisburse}`); // Debug disbursement eligibility
                
                const disburseBtnHtml = canDisburse 
                    ? `<button class="btn btn-sm btn-success" onclick="showDisburseLoanModalWithFees(${loan.loan_id}, '${loan.client_name}', '${loan.loan_type}', ${loan.disbursed_amount || loan.amount})">💸 Disburse</button>`
                    : `<button class="btn btn-sm" disabled style="background: #ccc;">⏳ Awaiting Fee</button>`;
                
                row.innerHTML = `
                    <td>#${loan.loan_id}</td>
                    <td>${loan.client_name}</td>
                    <td>${loan.client_phone || 'N/A'}</td>
                    <td>${loan.loan_type}</td>
                    <td>UGX ${requestedAmount}</td>
                    <td>UGX ${approvedAmount}</td>
                    <td>${approvedDate}</td>
                    <td>${loan.officer_name || 'N/A'}</td>
                    <td>${disburseBtnHtml}</td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading approved loans:', error);
    }
}

// Show disburse loan modal with processing fee calculation
async function showDisburseLoanModalWithFees(loanId, clientName, loanType, requestedAmount) {
    try {
        // Get disbursement breakdown
        const response = await apiCall(`/accountant/disbursement-amount/${loanId}`);
        
        if (response.success) {
            const { approved_amount, processing_fee, net_disbursement, client_repays } = response.data;
            
            document.getElementById('disburseLoanId').value = loanId;
            document.getElementById('disburseClientName').textContent = clientName;
            document.getElementById('disburseLoanType').textContent = loanType;
            document.getElementById('disburseApprovedAmount').textContent = `UGX ${Number(approved_amount).toLocaleString()}`;
            document.getElementById('disburseProcessingFee').textContent = `- UGX ${Number(processing_fee).toLocaleString()}`;
            document.getElementById('disburseNetAmount').textContent = `UGX ${Number(net_disbursement).toLocaleString()}`;
            document.getElementById('disburseRepayAmount').textContent = `UGX ${Number(client_repays).toLocaleString()}`;
            document.getElementById('disbursementMethod').value = '';
            
            // Store the calculated amounts in the form for submission
            const form = document.getElementById('disburseLoanForm');
            form.setAttribute('data-net-amount', net_disbursement);
            form.setAttribute('data-processing-fee', processing_fee);
            
            openModal('disburseLoanModal');
        }
    } catch (error) {
        alert('❌ Failed to calculate disbursement amount. ' + (error.message || ''));
    }
}

// Handle disbursement form submission
document.addEventListener('DOMContentLoaded', () => {
    const disburseLoanForm = document.getElementById('disburseLoanForm');
    if (disburseLoanForm) {
        disburseLoanForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const loanId = document.getElementById('disburseLoanId').value;
            const payment_method = document.getElementById('disbursementMethod').value;
            
            // Get the calculated amounts from data attributes
            const net_amount = parseFloat(disburseLoanForm.getAttribute('data-net-amount'));
            const processing_fee = parseFloat(disburseLoanForm.getAttribute('data-processing-fee'));
            
            if (!payment_method) {
                alert('Please select a disbursement method');
                return;
            }
            
            try {
                const response = await apiCall('/accountant/disburse-with-accounting', 'POST', {
                    loanId: parseInt(loanId),
                    disbursement_amount: net_amount,
                    payment_method,
                    processing_fee_deducted: processing_fee
                });
                
                if (response.success) {
                    alert('✅ Loan disbursed successfully! Processing fee recorded. Accounting entries posted.');
                    closeModal('disburseLoanModal');
                    loadApprovedLoansForDisbursement();
                    loadAccountBalances();
                    loadFinancialSummary();
                } else {
                    alert('❌ ' + (response.message || 'Failed to disburse loan'));
                }
            } catch (error) {
                console.error('Error disbursing loan:', error);
                alert('❌ Error: ' + (error.message || 'Failed to disburse loan'));
            }
        });
    }
    
    // Handle payment recording form
    const recordPaymentForm = document.getElementById('recordPaymentForm');
    if (recordPaymentForm) {
        recordPaymentForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const loanId = document.getElementById('paymentLoanId').value;
            const amount = document.getElementById('paymentAmount').value;
            const payment_method = document.getElementById('paymentMethodSelect').value;
            const receipt_number = document.getElementById('receiptNumber').value;
            const notes = document.getElementById('paymentNotes').value;
            
            if (!payment_method) {
                alert('Please select a payment method');
                return;
            }
            
            try {
                // Only accountants can record payments
                const response = await apiCall('/accountant/record-payment-with-accounting', 'POST', {
                    loanId: parseInt(loanId),
                    amount: parseFloat(amount),
                    payment_method,
                    receipt_number,
                    notes
                });
                
                if (response.success) {
                    const receiptNum = response.data.receiptNumber;
                    const printBtn = confirm(`✅ Payment recorded successfully!\n\nReceipt Number: ${receiptNum}\nNew Balance: UGX ${Number(response.data.newBalance).toLocaleString()}\n\nWould you like to print the receipt?`);
                    
                    if (printBtn) {
                        window.open(`/api/accountant/receipt/print/${receiptNum}`, '_blank');
                    }
                    
                    closeModal('recordPaymentModal');
                    // Reload the search if there was a search term
                    const searchTerm = document.getElementById('searchClientPayment').value;
                    if (searchTerm) {
                        searchClientForPayment();
                    }
                    loadAccountBalances();
                } else {
                    alert('❌ ' + (response.message || 'Failed to record payment'));
                }
            } catch (error) {
                console.error('Error recording payment:', error);
                alert('❌ Error: ' + (error.message || 'Failed to record payment'));
            }
        });
    }
});

// Update the showAccountantSection function to load data for new sections
const originalShowAccountantSection = showAccountantSection;
showAccountantSection = function(section) {
    originalShowAccountantSection.call(this, section);
    
    if (section === 'accounts') {
        loadAccountBalances();
    } else if (section === 'disbursements') {
        loadApprovedLoans();
    } else if (section === 'fees') {
        loadPendingApplicationFees();
    } else if (section === 'petty-cash') {
        loadPettyCash();
    } else if (section === 'transactions') {
        loadRecentTransactions();
    }
};

// Show payment history for a loan
async function showPaymentHistory(loanId) {
    try {
        const response = await apiCall(`/accountant/loan-payments/${loanId}`);
        
        if (response.success) {
            const { loan, payments } = response.data;
            
            // Populate loan info
            document.getElementById('historyLoanId').textContent = loan.id;
            document.getElementById('historyClientName').textContent = loan.client_name;
            document.getElementById('historyLoanBalance').textContent = `UGX ${Number(loan.balance).toLocaleString()}`;
            document.getElementById('historyLoanStatus').textContent = loan.status;
            
            // Populate payments table
            const tbody = document.getElementById('paymentHistoryTable');
            tbody.innerHTML = '';
            
            if (payments.length === 0) {
                tbody.innerHTML = '<tr><td colspan="7" class="text-center">No payments recorded yet</td></tr>';
            } else {
                payments.forEach(payment => {
                    const row = document.createElement('tr');
                    const paymentDate = new Date(payment.payment_date).toLocaleString();
                    row.innerHTML = `
                        <td>${paymentDate}</td>
                        <td><strong>UGX ${Number(payment.amount).toLocaleString()}</strong></td>
                        <td>${formatPaymentMethod(payment.payment_method)}</td>
                        <td>${payment.receipt_number || 'N/A'}</td>
                        <td>${payment.recorded_by_name || 'System'}</td>
                        <td>${payment.notes || '-'}</td>
                        <td>
                            <button class="btn btn-sm btn-warning" onclick="showEditPaymentModal(${payment.id}, ${loan.id}, ${payment.amount}, '${payment.payment_method}', '${payment.receipt_number || ''}', '${payment.notes || ''}')">✏️ Edit</button>
                            <button class="btn btn-sm btn-danger" onclick="deletePayment(${payment.id}, ${loan.id})">🗑️ Delete</button>
                        </td>
                    `;
                    tbody.appendChild(row);
                });
            }
            
            openModal('paymentHistoryModal');
        } else {
            alert('❌ Failed to load payment history');
        }
    } catch (error) {
        console.error('Error loading payment history:', error);
        alert('❌ Error loading payment history');
    }
}

// Format payment method for display
function formatPaymentMethod(method) {
    const methodLower = (method || '').toLowerCase();
    if (methodLower === 'cash') return '💵 Cash';
    if (methodLower === 'bank') return '🏦 Bank';
    if (methodLower === 'mobile_money') return '📱 Mobile Money';
    return method;
}

// Show edit payment modal
function showEditPaymentModal(paymentId, loanId, amount, method, receiptNumber, notes) {
    document.getElementById('editPaymentId').value = paymentId;
    document.getElementById('editPaymentLoanId').value = loanId;
    document.getElementById('editPaymentAmount').value = amount;
    document.getElementById('editPaymentMethod').value = method;
    document.getElementById('editReceiptNumber').value = receiptNumber;
    document.getElementById('editPaymentNotes').value = notes;
    
    closeModal('paymentHistoryModal');
    openModal('editPaymentModal');
}

// Handle edit payment form submission
document.addEventListener('DOMContentLoaded', () => {
    const editPaymentForm = document.getElementById('editPaymentForm');
    if (editPaymentForm && !editPaymentForm.hasAttribute('data-handler-attached')) {
        editPaymentForm.setAttribute('data-handler-attached', 'true');
        editPaymentForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const paymentId = document.getElementById('editPaymentId').value;
            const loanId = document.getElementById('editPaymentLoanId').value;
            const amount = document.getElementById('editPaymentAmount').value;
            const payment_method = document.getElementById('editPaymentMethod').value;
            const receipt_number = document.getElementById('editReceiptNumber').value;
            const notes = document.getElementById('editPaymentNotes').value;
            
            if (!payment_method) {
                alert('Please select a payment method');
                return;
            }
            
            try {
                const response = await apiCall(`/accountant/payment/${paymentId}`, 'PUT', {
                    amount: parseFloat(amount),
                    payment_method,
                    receipt_number,
                    notes
                });
                
                if (response.success) {
                    alert(`✅ Payment updated successfully!\nNew Balance: UGX ${Number(response.data.newBalance).toLocaleString()}`);
                    closeModal('editPaymentModal');
                    // Reload payment history
                    showPaymentHistory(parseInt(loanId));
                    // Reload search results if there's a search term
                    const searchTerm = document.getElementById('searchClientPayment').value;
                    if (searchTerm) {
                        searchClientForPayment();
                    }
                    loadAccountBalances();
                } else {
                    alert('❌ ' + (response.message || 'Failed to update payment'));
                }
            } catch (error) {
                console.error('Error updating payment:', error);
                alert('❌ Error: ' + (error.message || 'Failed to update payment'));
            }
        });
    }
});

// Delete a payment
async function deletePayment(paymentId, loanId) {
    if (!confirm('⚠️ Are you sure you want to delete this payment?\n\nThis will:\n- Remove the payment record\n- Increase the loan balance\n- Reverse all accounting entries\n- Notify the client')) {
        return;
    }
    
    try {
        const response = await apiCall(`/accountant/payment/${paymentId}`, 'DELETE');
        
        if (response.success) {
            alert(`✅ Payment deleted successfully!\nNew Balance: UGX ${Number(response.data.newBalance).toLocaleString()}`);
            // Reload payment history
            showPaymentHistory(loanId);
            // Reload search results if there's a search term
            const searchTerm = document.getElementById('searchClientPayment').value;
            if (searchTerm) {
                searchClientForPayment();
            }
            loadAccountBalances();
        } else {
            alert('❌ ' + (response.message || 'Failed to delete payment'));
        }
    } catch (error) {
        console.error('Error deleting payment:', error);
        alert('❌ Error: ' + (error.message || 'Failed to delete payment'));
    }
}

// Run alert check
async function runAlertCheck() {
    const button = event.target;
    button.disabled = true;
    button.textContent = '🔄 Checking...';
    
    try {
        const response = await apiCall('/accountant/check-overdue-loans');
        
        if (response.success) {
            const { overdue, dueSoon, upcomingThisWeek } = response.data;
            
            document.getElementById('alertOverdueCount').textContent = overdue.length;
            document.getElementById('alertDueSoonCount').textContent = dueSoon.length;
            document.getElementById('alertUpcomingCount').textContent = upcomingThisWeek.length;
            
            const totalAlerts = overdue.length + dueSoon.length + upcomingThisWeek.length;
            document.getElementById('alertCheckMessage').innerHTML = `
                <p style="margin: 0;">
                    ✅ Alert check complete! Found <strong>${totalAlerts}</strong> loans requiring attention.<br>
                    Notifications sent to officers and clients.
                </p>
            `;
            
            openModal('alertCheckModal');
        } else {
            alert('❌ Failed to check overdue loans');
        }
    } catch (error) {
        console.error('Error checking overdue loans:', error);
        alert('❌ Error checking overdue loans');
    } finally {
        button.disabled = false;
        button.textContent = '🔔 Check Overdue Loans & Send Alerts';
    }
}

// ============================================
// LOAN FEES MANAGEMENT
// ============================================

// Load pending application fees for confirmation
async function loadPendingApplicationFees() {
    try {
        const response = await apiCall('/accountant/pending-application-fees');
        if (response.success) {
            const tbody = document.getElementById('pendingFeesTable');
            tbody.innerHTML = '';
            
            if (response.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="7" class="text-center">No pending application fees</td></tr>';
                return;
            }
            
            response.data.forEach(fee => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${fee.client_name}</td>
                    <td>${fee.client_phone}</td>
                    <td>${fee.client_email || 'N/A'}</td>
                    <td>UGX ${Number(fee.amount).toLocaleString()}</td>
                    <td><span style="padding: 4px 8px; background: #dbeafe; border-radius: 4px;">${fee.payment_method}</span></td>
                    <td>${fee.transaction_reference || 'N/A'}</td>
                    <td>${new Date(fee.created_at).toLocaleDateString()}</td>
                    <td>
                        <button onclick="confirmApplicationFee(${fee.fee_id})" class="btn btn-success btn-sm">✅ Confirm</button>
                    </td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading pending fees:', error);
    }
}

// Confirm application fee payment
async function confirmApplicationFee(feeId) {
    if (!confirm('Confirm that you have received this application fee payment?')) {
        return;
    }
    
    try {
        const response = await apiCall('/accountant/confirm-application-fee', 'POST', {
            fee_id: feeId
        });
        
        if (response.success) {
            alert('✅ Application fee confirmed successfully!');
            loadPendingApplicationFees();
            loadFinancialSummary();
        }
    } catch (error) {
        alert('❌ Failed to confirm application fee. ' + (error.message || ''));
    }
}

// ============================================
// SEARCH/FILTER FUNCTIONS
// ============================================

// Filter clients table
function filterClientsTable() {
    const searchInput = document.getElementById('clientSearchInput');
    const filter = searchInput.value.toLowerCase();
    const table = document.getElementById('clientsTable');
    const rows = table.getElementsByTagName('tr');
    
    for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const cells = row.getElementsByTagName('td');
        
        if (cells.length > 0) {
            const name = cells[0].textContent || cells[0].innerText;
            const phone = cells[1].textContent || cells[1].innerText;
            const village = cells[2].textContent || cells[2].innerText;
            const officer = cells[3].textContent || cells[3].innerText;
            
            const searchText = (name + ' ' + phone + ' ' + village + ' ' + officer).toLowerCase();
            
            if (searchText.indexOf(filter) > -1) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        }
    }
}

// Filter users table
function filterUsersTable() {
    const searchInput = document.getElementById('userSearchInput');
    const filter = searchInput.value.toLowerCase();
    const table = document.getElementById('usersTable');
    const rows = table.getElementsByTagName('tr');
    
    for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const cells = row.getElementsByTagName('td');
        
        if (cells.length > 0) {
            const name = cells[0].textContent || cells[0].innerText;
            const email = cells[1].textContent || cells[1].innerText;
            const role = cells[2].textContent || cells[2].innerText;
            
            const searchText = (name + ' ' + email + ' ' + role).toLowerCase();
            
            if (searchText.indexOf(filter) > -1) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        }
    }
}

// Filter pending loans table
function filterPendingLoansTable() {
    const searchInput = document.getElementById('pendingLoansSearchInput');
    const filter = searchInput.value.toLowerCase();
    const table = document.getElementById('pendingLoansTable');
    const rows = table.getElementsByTagName('tr');
    
    for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const cells = row.getElementsByTagName('td');
        
        if (cells.length > 0) {
            const clientName = cells[0].textContent || cells[0].innerText;
            const loanType = cells[1].textContent || cells[1].innerText;
            const amount = cells[2].textContent || cells[2].innerText;
            const officer = cells[3].textContent || cells[3].innerText;
            
            const searchText = (clientName + ' ' + loanType + ' ' + amount + ' ' + officer).toLowerCase();
            
            if (searchText.indexOf(filter) > -1) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        }
    }
}

// Filter officer clients table
function filterOfficerClientsTable() {
    const searchInput = document.getElementById('officerClientsSearchInput');
    if (!searchInput) return;
    
    const filter = searchInput.value.toLowerCase();
    const table = document.getElementById('assignedClientsTable');
    const rows = table.getElementsByTagName('tr');
    
    for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const cells = row.getElementsByTagName('td');
        
        if (cells.length > 0) {
            const clientName = cells[0].textContent || cells[0].innerText;
            const phone = cells[1].textContent || cells[1].innerText;
            const loanType = cells[2].textContent || cells[2].innerText;
            const status = cells[4].textContent || cells[4].innerText;
            
            const searchText = (clientName + ' ' + phone + ' ' + loanType + ' ' + status).toLowerCase();
            
            if (searchText.indexOf(filter) > -1) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        }
    }
}

// Filter application fees table
function filterApplicationFeesTable() {
    const searchInput = document.getElementById('applicationFeesSearchInput');
    if (!searchInput) return;
    
    const filter = searchInput.value.toLowerCase();
    const table = document.getElementById('pendingFeesTable');
    const rows = table.getElementsByTagName('tr');
    
    for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const cells = row.getElementsByTagName('td');
        
        if (cells.length > 0) {
            const clientName = cells[0].textContent || cells[0].innerText;
            const phone = cells[1].textContent || cells[1].innerText;
            const loanType = cells[2].textContent || cells[2].innerText;
            const paymentMethod = cells[5].textContent || cells[5].innerText;
            
            const searchText = (clientName + ' ' + phone + ' ' + loanType + ' ' + paymentMethod).toLowerCase();
            
            if (searchText.indexOf(filter) > -1) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        }
    }
}

// Filter disbursements table
function filterDisbursementsTable() {
    const searchInput = document.getElementById('disbursementsSearchInput');
    if (!searchInput) return;
    
    const filter = searchInput.value.toLowerCase();
    const table = document.getElementById('approvedLoansTable');
    const rows = table.getElementsByTagName('tr');
    
    for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const cells = row.getElementsByTagName('td');
        
        if (cells.length > 0) {
            const loanId = cells[0].textContent || cells[0].innerText;
            const clientName = cells[1].textContent || cells[1].innerText;
            const phone = cells[2].textContent || cells[2].innerText;
            const loanType = cells[3].textContent || cells[3].innerText;
            
            const searchText = (loanId + ' ' + clientName + ' ' + phone + ' ' + loanType).toLowerCase();
            
            if (searchText.indexOf(filter) > -1) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        }
    }
}

// Filter transactions table
function filterTransactionsTable() {
    const searchInput = document.getElementById('transactionsSearchInput');
    if (!searchInput) return;
    
    const filter = searchInput.value.toLowerCase();
    const table = document.getElementById('transactionsTable');
    const rows = table.getElementsByTagName('tr');
    
    for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const cells = row.getElementsByTagName('td');
        
        if (cells.length > 0) {
            const type = cells[0].textContent || cells[0].innerText;
            const amount = cells[1].textContent || cells[1].innerText;
            const description = cells[2].textContent || cells[2].innerText;
            const creator = cells[4].textContent || cells[4].innerText;
            
            const searchText = (type + ' ' + amount + ' ' + description + ' ' + creator).toLowerCase();
            
            if (searchText.indexOf(filter) > -1) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        }
    }
}

// Filter petty cash table
function filterPettyCashTable() {
    const searchInput = document.getElementById('pettyCashSearchInput');
    if (!searchInput) return;
    
    const filter = searchInput.value.toLowerCase();
    const table = document.getElementById('pettyCashTable');
    const rows = table.getElementsByTagName('tr');
    
    for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const cells = row.getElementsByTagName('td');
        
        if (cells.length > 0) {
            const type = cells[0].textContent || cells[0].innerText;
            const amount = cells[1].textContent || cells[1].innerText;
            const description = cells[2].textContent || cells[2].innerText;
            
            const searchText = (type + ' ' + amount + ' ' + description).toLowerCase();
            
            if (searchText.indexOf(filter) > -1) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        }
    }
}

// View loan application details (for officers)
async function viewLoanApplication(loanId) {
    try {
        const response = await apiCall(`/officer/loan-details/${loanId}`);
        if (response.success && response.data) {
            const loan = response.data;
            
            const details = `
                <div style="background: white; padding: 20px; border-radius: 8px;">
                    <h3>📋 Loan Application Details</h3>
                    <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 15px; margin-top: 20px;">
                        <div><strong>Client Name:</strong> ${loan.client_name}</div>
                        <div><strong>Phone:</strong> ${loan.phone}</div>
                        <div><strong>Loan Type:</strong> ${loan.loan_type}</div>
                        <div><strong>Amount Requested:</strong> UGX ${Number(loan.amount).toLocaleString()}</div>
                        <div><strong>Purpose:</strong> ${loan.purpose}</div>
                        <div><strong>Duration:</strong> ${loan.duration_months} month(s)</div>
                        <div><strong>Repayment Mode:</strong> ${loan.repayment_mode}</div>
                        <div><strong>Interest Rate:</strong> ${loan.interest_rate}%</div>
                        <div><strong>Application Date:</strong> ${new Date(loan.created_at).toLocaleDateString()}</div>
                        <div><strong>Status:</strong> <span style="background: #fff3cd; padding: 5px 10px; border-radius: 5px; font-weight: bold;">${loan.status.toUpperCase()}</span></div>
                        ${loan.collateral_type ? `<div style="grid-column: 1 / -1;"><strong>Collateral:</strong> ${loan.collateral_type} ${loan.collateral_value ? '- UGX ' + Number(loan.collateral_value).toLocaleString() : ''}<br/>${loan.collateral_description || ''}</div>` : ''}
                    </div>
                    <div style="margin-top: 20px; padding: 15px; background: #e8f5e9; border-radius: 5px;">
                        <p><strong>💡 Next Steps:</strong></p>
                        <p>This application is pending manager approval. Once reviewed and approved by the manager, the accountant will disburse the funds.</p>
                    </div>
                </div>
            `;
            
            alert(details.replace(/<[^>]*>/g, '\n')); // Simple alert for now
            // TODO: Create a proper modal for this
        }
    } catch (error) {
        console.error('Error loading loan application:', error);
        alert('Failed to load loan application details');
    }
}

// ============================================
// LOAN APPLICATION PREVIEW & CALCULATION
// ============================================

// Preview loan application before submission
async function previewLoanApplication() {
    const form = document.getElementById('clientLoanApplicationForm');
    const amount = document.getElementById('loanAmount').value;
    const loanType = document.getElementById('loanType').value;
    const purpose = document.getElementById('loanPurpose').value;
    const duration = document.getElementById('loanDuration').value;
    const repaymentMode = document.getElementById('repaymentMode').value;
    
    // Validate required fields
    if (!amount || !loanType || !purpose || !duration || !repaymentMode) {
        alert('Please fill in all required fields before previewing');
        return;
    }

    // Check if client is eligible and has paid application fee
    try {
        const overview = await apiCall('/client/loan-overview');
        if (overview && overview.eligibleForTopup) {
            // If eligible for top-up or new loan, check fee status
            const feeStatus = overview.data?.application_fee_status || 'pending';
            if (feeStatus !== 'paid') {
                alert('You must pay the application fee before applying for a new or top-up loan. Please pay at the office or via mobile money, then try again.');
                return;
            }
        }
    } catch (err) {
        console.error('Error checking application fee status:', err);
    }
    
    // Calculate preview
    await calculateLoanPreview();
    
    // Show preview and hide form fields
    document.getElementById('loanPreview').style.display = 'block';
    
    // Disable form inputs
    const inputs = form.querySelectorAll('input, select, textarea');
    inputs.forEach(input => input.disabled = true);
    
    // Change submit button text
    document.getElementById('submitLoanBtn').textContent = '✅ Confirm & Submit Application';
    
    // Scroll to preview
    document.getElementById('loanPreview').scrollIntoView({ behavior: 'smooth', block: 'center' });
}

// Update available repayment modes based on loan duration
function updateRepaymentModes() {
    const duration = parseFloat(document.getElementById('loanDuration').value) || 0;
    const repaymentModeSelect = document.getElementById('repaymentMode');
    
    if (!repaymentModeSelect) return;
    
    // Get all options
    const dailyOption = repaymentModeSelect.querySelector('option[value="daily"]');
    const weeklyOption = repaymentModeSelect.querySelector('option[value="weekly"]');
    const biWeeklyOption = repaymentModeSelect.querySelector('option[value="bi-weekly"]');
    const monthlyOption = repaymentModeSelect.querySelector('option[value="monthly"]');
    
    // Reset all options
    if (dailyOption) dailyOption.disabled = false;
    if (weeklyOption) weeklyOption.disabled = false;
    if (biWeeklyOption) biWeeklyOption.disabled = false;
    if (monthlyOption) monthlyOption.disabled = false;
    
    // Enable/disable based on duration
    if (duration === 0.25) {
        // 1 Week - only daily and weekly make sense
        if (biWeeklyOption) biWeeklyOption.disabled = true;
        if (monthlyOption) monthlyOption.disabled = true;
    } else if (duration === 0.5) {
        // 2 Weeks - daily, weekly, bi-weekly
        if (monthlyOption) monthlyOption.disabled = true;
    } else if (duration >= 1) {
        // 1 Month or more - all modes available
        // No restrictions
    }
    
    // If currently selected option is disabled, reset to empty
    const currentValue = repaymentModeSelect.value;
    const currentOption = repaymentModeSelect.querySelector(`option[value="${currentValue}"]`);
    if (currentOption && currentOption.disabled) {
        repaymentModeSelect.value = '';
    }
}

// Calculate loan preview
async function calculateLoanPreview() {
    const amount = parseFloat(document.getElementById('loanAmount').value) || 0;
    const loanType = document.getElementById('loanType').value;
    const duration = parseFloat(document.getElementById('loanDuration').value) || 1;
    const repaymentMode = document.getElementById('repaymentMode').value;
    
    if (amount === 0 || !repaymentMode) {
        document.getElementById('loanPreview').style.display = 'none';
        return;
    }
    
    // Default rates (should match backend)
    const interestRate = 20; // 20% FLAT rate
    const processingFee = 0;
    const applicationFee = 5000;
    
    // Calculate interest using FLAT rate formula: Balance = Amount × (1 + Rate/100)
    const interestAmount = (amount * interestRate) / 100;
    const totalBalance = amount + interestAmount; // What client repays
    const totalRepayable = totalBalance; // For display
    
    // Calculate installment based on repayment mode
    let installments = duration; // default monthly
    let modeText = 'Monthly';
    
    if (repaymentMode === 'daily') {
        installments = duration * 30; // approximate days per month
        modeText = 'Daily';
    } else if (repaymentMode === 'weekly') {
        installments = duration * 4; // approximate weeks per month
        modeText = 'Weekly';
    } else if (repaymentMode === 'bi-weekly') {
        installments = duration * 2; // bi-weekly
        modeText = 'Bi-Weekly';
    } else {
        modeText = 'Monthly';
    }
    
    const installmentAmount = totalBalance / installments;
    
    // Update preview
    document.getElementById('previewAmount').textContent = `UGX ${amount.toLocaleString()}`;
    document.getElementById('previewInterestRate').textContent = `${interestRate}% FLAT`;
    document.getElementById('previewProcessingFee').textContent = `UGX ${processingFee.toLocaleString()}`;
    document.getElementById('previewApplicationFee').textContent = `UGX ${applicationFee.toLocaleString()}`;
    document.getElementById('previewInterest').textContent = `UGX ${interestAmount.toLocaleString('en-US', {maximumFractionDigits: 0})}`;
    document.getElementById('previewTotal').textContent = `UGX ${totalRepayable.toLocaleString('en-US', {maximumFractionDigits: 0})}`;
    document.getElementById('previewInstallment').textContent = `UGX ${installmentAmount.toLocaleString('en-US', {maximumFractionDigits: 0})}`;
    document.getElementById('previewModeText').textContent = modeText;
    
    // Show preview section
    document.getElementById('loanPreview').style.display = 'block';
}

// Reset loan form
function resetLoanForm() {
    document.getElementById('loanPreview').style.display = 'none';
    const form = document.getElementById('clientLoanApplicationForm');
    const inputs = form.querySelectorAll('input, select, textarea');
    inputs.forEach(input => input.disabled = false);
    document.getElementById('submitLoanBtn').textContent = '📤 Submit Loan Application';
}



