const express = require('express');
const bodyParser = require('body-parser');
const cors = require('cors');
const dotenv = require('dotenv');
const path = require('path');
const authRoutes = require('./routes/auth');
const managerRoutes = require('./routes/manager');
const officerRoutes = require('./routes/officer');
const clientRoutes = require('./routes/client');
const accountantRoutes = require('./routes/accountant');
const enrollmentRoutes = require('./routes/enrollment');
const loanApplicationRoutes = require('./routes/loanApplication');
const financialReportsRoutes = require('./routes/financialReports');
const { connectDB } = require('./db');

// Load environment variables from .env file
dotenv.config({ path: path.join(__dirname, '.env') });

// Set JWT_SECRET with fallback if not found in .env
if (!process.env.JWT_SECRET) {
    console.warn('⚠️  WARNING: JWT_SECRET not found in .env file. Using fallback value.');
    process.env.JWT_SECRET = 'your_super_secret_jwt_key_change_this_in_production_12345';
}

console.log('✅ JWT_SECRET loaded:', process.env.JWT_SECRET ? 'Yes' : 'No');

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(cors());
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));

// Serve static files from public directory
app.use(express.static(path.join(__dirname, 'public')));

// Serve uploaded files
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));

// Database connection
connectDB();

// Routes
app.use('/api/auth', authRoutes);
app.use('/api/manager', managerRoutes);
app.use('/api/officer', officerRoutes);
app.use('/api/client', clientRoutes);
app.use('/api/accountant', accountantRoutes);
app.use('/api/enrollment', enrollmentRoutes);
app.use('/api/loan-application', loanApplicationRoutes);
app.use('/api/financial-reports', financialReportsRoutes);

// 404 handler for API routes
app.use('/api/*', (req, res) => {
    res.status(404).json({ success: false, error: 'API endpoint not found' });
});

// Serve index.html for root route
app.get('/', (req, res) => {
    res.sendFile(path.join(__dirname, 'public/index.html'));
});

// Global error handler
app.use((err, req, res, next) => {
    console.error('Global error handler:', err);
    res.status(500).json({ success: false, error: err.message || 'Internal server error' });
});

// Start server
app.listen(PORT, () => {
    console.log(`🚀 Server is running on http://localhost:${PORT}`);
});