const { pool, dbGet, dbAll, dbRun } = require('../db');

// Get assigned clients
const getAssignedClients = async (req, res) => {
    const officerId = req.user.id;

    try {
        const clients = await dbAll(
            `SELECT 
                u.id as client_id, u.name as client_name, u.phone, u.email,
                l.id as loan_id, l.loan_type, l.amount, l.balance, 
                l.next_payment_date, l.status, l.amount_paid, l.created_at,
                l.purpose, l.duration_months, l.repayment_mode,
                CASE 
                    WHEN l.status = 'pending' THEN '⏳ New Application'
                    WHEN l.status = 'approved' THEN '✅ Approved - Pending Disbursement'
                    WHEN l.next_payment_date < date('now') AND l.status = 'disbursed' THEN 'Overdue'
                    WHEN l.next_payment_date = date('now') AND l.status = 'disbursed' THEN 'Due Today'
                    WHEN l.status = 'disbursed' THEN 'Active'
                    WHEN l.status = 'active' THEN 'Active'
                    WHEN l.status = 'overdue' THEN 'Overdue'
                    WHEN l.status = 'completed' THEN 'Completed'
                    WHEN l.status = 'rejected' THEN 'Rejected'
                    ELSE 'No Loan'
                END as payment_status
            FROM users u
            LEFT JOIN loans l ON u.id = l.user_id 
            WHERE u.officer_id = ? AND u.role = 'client'
            ORDER BY 
                CASE 
                    WHEN l.status = 'pending' THEN 1
                    WHEN l.status = 'approved' THEN 2
                    WHEN l.status IN ('disbursed', 'active', 'overdue') THEN 3
                    ELSE 4
                END,
                l.created_at DESC`,
            [officerId]
        );

        res.json({ success: true, data: clients });
    } catch (error) {
        console.error('Error fetching assigned clients:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get daily follow-up tasks
const getDailyTasks = async (req, res) => {
    const officerId = req.user.id;

    try {
        // Get clients with payments due today or overdue
        const tasks = await dbAll(
            `SELECT 
                u.id as client_id, u.name as client_name, u.phone,
                l.id as loan_id, l.loan_type, l.amount, l.balance, 
                l.next_payment_date,
                CAST(julianday('now') - julianday(l.next_payment_date) AS INTEGER) as days_overdue
            FROM users u
            JOIN loans l ON u.id = l.user_id
            WHERE u.officer_id = ? 
            AND l.status IN ('disbursed', 'active')
            AND l.next_payment_date <= date('now')
            ORDER BY l.next_payment_date ASC`,
            [officerId]
        );

        // Count overdue clients
        const overdueResult = await dbGet(
            `SELECT COUNT(DISTINCT u.id) as count
            FROM users u
            JOIN loans l ON u.id = l.user_id
            WHERE u.officer_id = ? 
            AND l.status IN ('disbursed', 'active')
            AND l.next_payment_date < date('now')`,
            [officerId]
        );

        res.json({ 
            success: true, 
            data: {
                tasks: tasks || [],
                overdueCount: overdueResult ? overdueResult.count : 0
            }
        });
    } catch (error) {
        console.error('Error fetching daily tasks:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get detailed client information
const getClientDetails = async (req, res) => {
    const { clientId } = req.params;
    const officerId = req.user.id;

    try {
        // Get client info
        const client = await dbGet(
            `SELECT id, name, email, phone, national_id, village, parish, division, 
                    occupation, photo_url, created_at
            FROM users 
            WHERE id = ? AND officer_id = ? AND role = 'client'`,
            [clientId, officerId]
        );

        if (!client) {
            return res.status(404).json({ success: false, message: 'Client not found or not assigned to you' });
        }

        // Get guarantors for this client - handle if table doesn't exist
        let guarantors = [];
        try {
            guarantors = await dbAll(
                `SELECT id, name, phone, email, relationship, address, national_id, created_at
                FROM guarantors
                WHERE client_id = ?
                ORDER BY id`,
                [clientId]
            );
        } catch (guarantorError) {
            console.log('Guarantors query failed (table might not exist):', guarantorError.message);
            // Try alternative: get from loan_applications table if it exists
            try {
                const loanApp = await dbGet(
                    `SELECT guarantor1_name, guarantor1_phone, guarantor1_email, guarantor1_relationship,
                            guarantor2_name, guarantor2_phone, guarantor2_email, guarantor2_relationship
                    FROM loan_applications WHERE user_id = ? LIMIT 1`,
                    [clientId]
                );
                
                if (loanApp && loanApp.guarantor1_name) {
                    guarantors.push({
                        id: 1,
                        name: loanApp.guarantor1_name,
                        phone: loanApp.guarantor1_phone,
                        email: loanApp.guarantor1_email,
                        relationship: loanApp.guarantor1_relationship
                    });
                }
                if (loanApp && loanApp.guarantor2_name) {
                    guarantors.push({
                        id: 2,
                        name: loanApp.guarantor2_name,
                        phone: loanApp.guarantor2_phone,
                        email: loanApp.guarantor2_email,
                        relationship: loanApp.guarantor2_relationship
                    });
                }
            } catch (altError) {
                console.log('Alternative guarantor query also failed:', altError.message);
            }
        }

        // Get loan details
        const loans = await dbAll(
            `SELECT * FROM loans WHERE user_id = ? ORDER BY created_at DESC`,
            [clientId]
        );

        // Get repayment history - handle if table doesn't exist
        let repayments = [];
        try {
            repayments = await dbAll(
                `SELECT r.* FROM repayments r
                JOIN loans l ON r.loan_id = l.id
                WHERE l.user_id = ?
                ORDER BY r.payment_date DESC
                LIMIT 20`,
                [clientId]
            );
        } catch (repaymentError) {
            console.log('Repayments query failed (table might not exist):', repaymentError.message);
            // If repayments table doesn't exist, just return empty array
            repayments = [];
        }

        res.json({
            success: true,
            data: {
                client: client,
                guarantors: guarantors || [],
                loans: loans || [],
                repaymentHistory: repayments || []
            }
        });
    } catch (error) {
        console.error('Error fetching client details:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Generate report
const generateReport = async (req, res) => {
    const officerId = req.user.id;
    const { period } = req.query; // 'week' or 'month'

    try {
        const dateFilter = period === 'month' 
            ? "date('now', '-1 month')"
            : "date('now', '-7 days')";

        // Try to generate report with repayments table
        let report;
        try {
            report = await dbGet(
                `SELECT 
                    COUNT(DISTINCT l.user_id) as total_clients,
                    COUNT(r.id) as total_payments,
                    COALESCE(SUM(r.amount), 0) as total_collected,
                    COUNT(CASE WHEN l.next_payment_date < date('now') THEN 1 END) as overdue_loans
                FROM loans l
                LEFT JOIN repayments r ON l.id = r.loan_id AND r.payment_date >= ${dateFilter}
                JOIN users u ON l.user_id = u.id
                WHERE u.officer_id = ?`,
                [officerId]
            );
        } catch (repaymentError) {
            console.log('Report query with repayments failed (table might not exist):', repaymentError.message);
            // If repayments table doesn't exist, generate report without it
            report = await dbGet(
                `SELECT 
                    COUNT(DISTINCT l.user_id) as total_clients,
                    0 as total_payments,
                    0 as total_collected,
                    COUNT(CASE WHEN l.next_payment_date < date('now') THEN 1 END) as overdue_loans
                FROM loans l
                JOIN users u ON l.user_id = u.id
                WHERE u.officer_id = ?`,
                [officerId]
            );
        }

        res.json({ success: true, data: report });
    } catch (error) {
        console.error('Error generating report:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get follow-up tasks for officer
const getFollowUpTasks = async (req, res) => {
    const officerId = req.user.id;
    
    try {
        // Get loans requiring follow-up - handle if repayments table doesn't exist
        let tasks;
        try {
            tasks = await dbAll(`
                SELECT 
                    l.id as loan_id,
                    l.balance,
                    l.next_payment_date,
                    l.status,
                    l.repayment_mode,
                    u.id as client_id,
                    u.name as client_name,
                    u.phone as client_phone,
                    u.village,
                    CAST(julianday('now') - julianday(l.next_payment_date) AS INTEGER) as days_overdue,
                    CAST(julianday('now') - julianday(MAX(r.payment_date)) AS INTEGER) as days_since_last_payment
                FROM loans l
                INNER JOIN users u ON l.user_id = u.id
                LEFT JOIN repayments r ON l.id = r.loan_id
                WHERE u.officer_id = ?
                AND l.status IN ('disbursed', 'active', 'overdue')
                AND l.balance > 0
                GROUP BY l.id, u.id
                ORDER BY 
                    CASE 
                        WHEN l.status = 'overdue' THEN 1
                        WHEN CAST(julianday('now') - julianday(l.next_payment_date) AS INTEGER) >= -1 THEN 2
                        ELSE 3
                    END,
                    l.next_payment_date ASC
            `, [officerId]);
        } catch (repaymentError) {
            console.log('Follow-up tasks query with repayments failed (table might not exist):', repaymentError.message);
            // If repayments table doesn't exist, query without it
            tasks = await dbAll(`
                SELECT 
                    l.id as loan_id,
                    l.balance,
                    l.next_payment_date,
                    l.status,
                    l.repayment_mode,
                    u.id as client_id,
                    u.name as client_name,
                    u.phone as client_phone,
                    u.village,
                    CAST(julianday('now') - julianday(l.next_payment_date) AS INTEGER) as days_overdue,
                    NULL as days_since_last_payment
                FROM loans l
                INNER JOIN users u ON l.user_id = u.id
                WHERE u.officer_id = ?
                AND l.status IN ('disbursed', 'active', 'overdue')
                AND l.balance > 0
                ORDER BY 
                    CASE 
                        WHEN l.status = 'overdue' THEN 1
                        WHEN CAST(julianday('now') - julianday(l.next_payment_date) AS INTEGER) >= -1 THEN 2
                        ELSE 3
                    END,
                    l.next_payment_date ASC
            `, [officerId]);
        }
        
        const categorized = {
            overdue: [],
            dueToday: [],
            dueTomorrow: [],
            upcoming: []
        };
        
        tasks.forEach(task => {
            const daysOverdue = task.days_overdue || 0;
            
            if (daysOverdue > 0) {
                categorized.overdue.push({
                    ...task,
                    priority: daysOverdue > 7 ? 'high' : daysOverdue > 3 ? 'medium' : 'low',
                    action: 'Follow up urgently - loan is overdue'
                });
            } else if (daysOverdue === 0) {
                categorized.dueToday.push({
                    ...task,
                    priority: 'high',
                    action: 'Contact client about payment due today'
                });
            } else if (daysOverdue === -1) {
                categorized.dueTomorrow.push({
                    ...task,
                    priority: 'medium',
                    action: 'Remind client about payment due tomorrow'
                });
            } else if (daysOverdue >= -7) {
                categorized.upcoming.push({
                    ...task,
                    priority: 'low',
                    action: 'Upcoming payment this week'
                });
            }
        });
        
        res.json({ success: true, data: categorized });
    } catch (error) {
        console.error('Error fetching follow-up tasks:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get loan details for officer
const getLoanDetails = async (req, res) => {
    const { loanId } = req.params;
    const officerId = req.user.id;

    try {
        // Get loan with client info, ensure officer owns this client
        const loan = await dbGet(
            `SELECT 
                l.*,
                u.name as client_name, u.phone, u.email, u.national_id
            FROM loans l
            JOIN users u ON l.user_id = u.id
            WHERE l.id = ? AND u.officer_id = ?`,
            [loanId, officerId]
        );

        if (!loan) {
            return res.status(404).json({ success: false, message: 'Loan not found or not assigned to you' });
        }

        res.json({ success: true, data: loan });
    } catch (error) {
        console.error('Error fetching loan details:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get notifications
const getNotifications = async (req, res) => {
    const officerId = req.user.id;

    try {
        const notifications = await dbAll(
            `SELECT * FROM notifications 
             WHERE user_id = ? 
             ORDER BY created_at DESC 
             LIMIT 50`,
            [officerId]
        );

        res.json({ success: true, data: notifications });
    } catch (error) {
        console.error('Error fetching officer notifications:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Mark notification as read
const markNotificationRead = async (req, res) => {
    const { notificationId } = req.params;
    const officerId = req.user.id;

    try {
        // Verify notification belongs to officer
        const notification = await dbGet(
            'SELECT * FROM notifications WHERE id = ? AND user_id = ?',
            [notificationId, officerId]
        );

        if (!notification) {
            return res.status(404).json({ success: false, message: 'Notification not found' });
        }

        await dbRun(
            'UPDATE notifications SET is_read = 1 WHERE id = ?',
            [notificationId]
        );

        res.json({ success: true, message: 'Notification marked as read' });
    } catch (error) {
        console.error('Error marking notification as read:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

module.exports = {
    getAssignedClients,
    getDailyTasks,
    getClientDetails,
    generateReport,
    getFollowUpTasks,
    getLoanDetails,
    getNotifications,
    markNotificationRead
};