const { dbAll, dbGet, dbRun } = require('../db');

// Get financial oversight data for manager
const getFinancialOversight = async (req, res) => {
    try {
        // Get account balances - with fallback if tables don't exist
        let accountBalances = [];
        try {
            accountBalances = await dbAll(`
                SELECT 
                    account_code,
                    account_name,
                    COALESCE(balance, 0) as balance
                FROM chart_of_accounts
                WHERE account_code IN ('1110', '1120', '1130', '1140', '1210')
                ORDER BY account_code
            `);
        } catch (error) {
            console.log('Chart of accounts query failed:', error.message);
            // Fallback: Create default account structure
            accountBalances = [
                { account_code: '1110', account_name: 'Cash on Hand', balance: 0 },
                { account_code: '1120', account_name: 'Bank Account', balance: 0 },
                { account_code: '1130', account_name: 'Mobile Money', balance: 0 },
                { account_code: '1140', account_name: 'Petty Cash', balance: 0 },
                { account_code: '1210', account_name: 'Loan Receivables', balance: 0 }
            ];
        }

        // Get petty cash status from balance table (shared with accountant)
        let pettyCash = { allocated_amount: 0, current_balance: 0 };
        try {
            const pettyCashResult = await dbGet(`SELECT * FROM petty_cash_balance ORDER BY id DESC LIMIT 1`);
            if (pettyCashResult) {
                pettyCash = pettyCashResult;
            }
        } catch (error) {
            console.log('Petty cash balance query failed:', error.message);
        }

        // Get total loans disbursed and outstanding
        let loanStats = { total_loans: 0, total_disbursed: 0, outstanding_balance: 0 };
        try {
            const stats = await dbGet(`
                SELECT 
                    COUNT(*) as total_loans,
                    COALESCE(SUM(CASE WHEN status IN ('active', 'disbursed') THEN amount ELSE 0 END), 0) as total_disbursed,
                    COALESCE(SUM(CASE WHEN status IN ('active', 'disbursed') THEN balance ELSE 0 END), 0) as outstanding_balance
                FROM loans
            `);
            if (stats) loanStats = stats;
        } catch (error) {
            console.log('Loan stats query failed:', error.message);
        }

        // Get petty cash transaction history
        let pettyCashHistory = [];
        try {
            pettyCashHistory = await dbAll(`
                SELECT 
                    pc.transaction_date as date,
                    'Petty Cash' as account,
                    CASE WHEN pc.amount > 0 THEN 'allocation' ELSE 'expense' END as action,
                    ABS(pc.amount) as amount,
                    pc.description,
                    u.name as recorded_by
                FROM petty_cash pc
                LEFT JOIN users u ON pc.recorded_by = u.id
                ORDER BY pc.transaction_date DESC
                LIMIT 20
            `);
        } catch (error) {
            console.log('Petty cash history query failed:', error.message);
        }

        // Calculate totals
        const totalAllocated = pettyCash.allocated_amount || 0;
        const totalAvailable = pettyCash.current_balance || 0;
        const totalUsed = totalAllocated - totalAvailable;
        const utilizationRate = totalAllocated > 0 ? ((totalUsed / totalAllocated) * 100).toFixed(2) : 0;

        res.json({
            success: true,
            data: {
                accountBalances: accountBalances, // Cash on Hand, Bank, Mobile Money, Petty Cash, Loan Receivables
                pettyCash: {
                    allocated_amount: pettyCash.allocated_amount || 0,
                    current_balance: pettyCash.current_balance || 0,
                    spent: (pettyCash.allocated_amount || 0) - (pettyCash.current_balance || 0)
                },
                loanStats: {
                    totalLoans: loanStats.total_loans,
                    totalDisbursed: loanStats.total_disbursed,
                    outstandingBalance: loanStats.outstanding_balance
                },
                allocationHistory: pettyCashHistory,
                summary: {
                    totalAllocated,
                    totalAvailable,
                    totalUsed,
                    utilizationRate
                }
            }
        });
    } catch (error) {
        console.error('Error fetching financial oversight:', error);
        res.status(500).json({ success: false, message: error.message });
    }
};

// Get dashboard summary statistics
const getDashboardStats = async (req, res) => {
    try {
        // Total Clients
        const clientCount = await dbGet(
            "SELECT COUNT(*) as count FROM users WHERE role = 'client'"
        );

        // Total Loans Disbursed
        const loansDisbursed = await dbGet(
            "SELECT COUNT(*) as count, COALESCE(SUM(amount), 0) as total FROM loans WHERE status IN ('disbursed', 'completed', 'overdue', 'active')"
        );

        // Loans Pending Approval
        const pendingLoans = await dbGet(
            "SELECT COUNT(*) as count FROM loans WHERE status = 'pending'"
        );

        // Active Credit Officers
        const officerCount = await dbGet(
            "SELECT COUNT(*) as count FROM users WHERE role = 'officer'"
        );

        // Portfolio Overview - Current outstanding balances
        const portfolioData = await dbGet(
            `SELECT 
                COALESCE(SUM(balance), 0) as total_portfolio,
                COALESCE(SUM(amount), 0) as total_loaned,
                COALESCE(SUM(amount_paid), 0) as total_collected
            FROM loans WHERE status IN ('disbursed', 'overdue', 'active')`
        );

        // Calculate portfolio growth (comparing this month vs last month)
        const growthData = await dbGet(
            `SELECT 
                COALESCE(SUM(CASE WHEN strftime('%m', created_at) = strftime('%m', 'now') THEN amount ELSE 0 END), 0) as current_month,
                COALESCE(SUM(CASE WHEN strftime('%m', created_at) = strftime('%m', date('now', '-1 month')) THEN amount ELSE 0 END), 0) as last_month
            FROM loans WHERE status IN ('disbursed', 'completed', 'overdue', 'active')`
        );

        const growthPercentage = growthData.last_month > 0 
            ? ((growthData.current_month - growthData.last_month) / growthData.last_month * 100).toFixed(2)
            : (growthData.current_month > 0 ? 100 : 0);

        // Monthly loan performance (last 6 months)
        const monthlyPerformance = await dbAll(
            `SELECT 
                strftime('%Y-%m', created_at) as month,
                COUNT(*) as loan_count,
                COALESCE(SUM(amount), 0) as total_amount
            FROM loans 
            WHERE created_at >= date('now', '-6 months')
            AND status IN ('disbursed', 'completed', 'overdue', 'active')
            GROUP BY strftime('%Y-%m', created_at)
            ORDER BY month ASC`
        );

        res.json({
            success: true,
            data: {
                totalClients: clientCount.count,
                totalLoansDisbursed: loansDisbursed.count,
                totalDisbursedAmount: loansDisbursed.total,
                pendingApprovals: pendingLoans.count,
                activeCreditOfficers: officerCount.count,
                portfolio: {
                    totalValue: portfolioData.total_portfolio,
                    totalLoaned: portfolioData.total_loaned,
                    totalCollected: portfolioData.total_collected,
                    growthPercentage: growthPercentage
                },
                monthlyPerformance: monthlyPerformance
            }
        });
    } catch (error) {
        console.error('Error fetching dashboard stats:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get pending loan applications
const getPendingLoans = async (req, res) => {
    try {
        const loans = await dbAll(
            `SELECT 
                l.id, l.loan_type, l.amount, l.created_at,
                u.name as client_name, u.email as client_email, u.lc1_letter_url,
                o.name as officer_name
            FROM loans l
            JOIN users u ON l.user_id = u.id
            LEFT JOIN users o ON l.officer_id = o.id
            WHERE l.status = 'pending'
            ORDER BY l.created_at DESC`
        );

        res.json({ success: true, data: loans });
    } catch (error) {
        console.error('Error fetching pending loans:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Approve or reject loan
const updateLoanStatus = async (req, res) => {
    const { loanId } = req.params;
    const { status, officer_id, disbursed_amount } = req.body;
    const managerId = req.user.id;

    try {
        if (!['approved', 'rejected'].includes(status)) {
            return res.status(400).json({ success: false, message: 'Invalid status' });
        }

        // Get loan details first
        const loan = await dbGet('SELECT * FROM loans WHERE id = ?', [loanId]);
        
        if (!loan) {
            return res.status(404).json({ success: false, message: 'Loan not found' });
        }

        // Update loan with disbursed amount if provided
        const actualDisbursedAmount = disbursed_amount || loan.amount;
        
        await dbRun(
            `UPDATE loans 
            SET status = ?, 
                approved_by = ?, 
                approved_at = datetime('now'), 
                officer_id = ?,
                disbursed_amount = ?
            WHERE id = ?`,
            [status, managerId, officer_id || null, actualDisbursedAmount, loanId]
        );
        
        const message = status === 'approved' 
            ? `Your loan application has been approved! Requested: UGX ${loan.amount.toLocaleString()}, Approved: UGX ${Number(actualDisbursedAmount).toLocaleString()}`
            : `Your loan application has been rejected.`;
        
        await dbRun(
            'INSERT INTO notifications (user_id, message, type, created_at) VALUES (?, ?, ?, datetime(\'now\'))',
            [loan.user_id, message, 'approval']
        );

        res.json({ success: true, message: `Loan ${status} successfully` });
    } catch (error) {
        console.error('Error updating loan status:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get loan settings
const getLoanSettings = async (req, res) => {
    try {
        const settings = await dbAll('SELECT * FROM loan_settings ORDER BY loan_type');
        res.json({ success: true, data: settings });
    } catch (error) {
        console.error('Error fetching loan settings:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Update loan settings
const updateLoanSettings = async (req, res) => {
    const { loan_type, interest_rate, penalty_rate, early_payment_bonus } = req.body;

    console.log('Updating loan settings:', { loan_type, interest_rate, penalty_rate, early_payment_bonus });

    try {
        const result = await dbRun(
            `UPDATE loan_settings 
            SET interest_rate = ?, penalty_rate = ?, early_payment_bonus = ?, updated_at = datetime('now')
            WHERE loan_type = ?`,
            [interest_rate, penalty_rate, early_payment_bonus, loan_type]
        );

        console.log('Update result:', result);

        if (result.changes === 0) {
            return res.status(404).json({ success: false, message: 'Loan type not found' });
        }

        res.json({ success: true, message: 'Settings updated successfully' });
    } catch (error) {
        console.error('Error updating loan settings:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Get system settings (processing fee, etc.)
const getSystemSettings = async (req, res) => {
    try {
        const settings = await dbAll('SELECT * FROM system_settings');
        
        // Convert to key-value object for easier frontend use
        const settingsObj = {};
        settings.forEach(setting => {
            settingsObj[setting.setting_key] = {
                value: setting.setting_value,
                description: setting.description,
                updated_at: setting.updated_at
            };
        });
        
        res.json({ success: true, data: settingsObj });
    } catch (error) {
        console.error('Error fetching system settings:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Update system settings
const updateSystemSettings = async (req, res) => {
    const { setting_key, setting_value } = req.body;
    const manager_id = req.user.user_id;

    try {
        await dbRun(
            `UPDATE system_settings 
            SET setting_value = ?, updated_at = datetime('now'), updated_by = ?
            WHERE setting_key = ?`,
            [setting_value, manager_id, setting_key]
        );

        res.json({ success: true, message: 'System setting updated successfully' });
    } catch (error) {
        console.error('Error updating system setting:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get all users
const getAllUsers = async (req, res) => {
    try {
        const users = await dbAll(
            `SELECT id, name, email, phone, role, created_at 
            FROM users 
            WHERE role IN ('officer', 'client', 'accountant')
            ORDER BY role, name`
        );

        res.json({ success: true, data: users });
    } catch (error) {
        console.error('Error fetching users:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Create new user
const createUser = async (req, res) => {
    const { name, email, password, phone, role } = req.body;
    const bcrypt = require('bcryptjs');

    try {
        const existing = await dbGet('SELECT id FROM users WHERE email = ?', [email]);
        if (existing) {
            return res.status(400).json({ success: false, message: 'Email already exists' });
        }

        const hashedPassword = await bcrypt.hash(password, 10);

        await dbRun(
            'INSERT INTO users (name, email, password, phone, role) VALUES (?, ?, ?, ?, ?)',
            [name, email, hashedPassword, phone, role]
        );

        res.json({ success: true, message: 'User created successfully' });
    } catch (error) {
        console.error('Error creating user:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Reset password (Admin function - reset any user's password)
const resetPassword = async (req, res) => {
    const { userId } = req.params;
    const { newPassword } = req.body;
    const bcrypt = require('bcryptjs');

    try {
        const hashedPassword = await bcrypt.hash(newPassword, 10);
        await dbRun('UPDATE users SET password = ? WHERE id = ?', [hashedPassword, userId]);

        res.json({ success: true, message: 'Password reset successfully' });
    } catch (error) {
        console.error('Error resetting password:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Toggle user status (activate/deactivate)
const updateUserStatus = async (req, res) => {
    const { userId } = req.params;
    const { status } = req.body;

    try {
        // Validate status
        if (!['active', 'inactive'].includes(status)) {
            return res.status(400).json({ success: false, message: 'Invalid status value' });
        }

        await dbRun('UPDATE users SET status = ? WHERE id = ?', [status, userId]);

        res.json({ success: true, message: `User ${status === 'active' ? 'activated' : 'deactivated'} successfully` });
    } catch (error) {
        console.error('Error updating user status:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Change own password (requires current password)
const changePassword = async (req, res) => {
    const managerId = req.user.id;
    const { currentPassword, newPassword } = req.body;
    const bcrypt = require('bcryptjs');

    try {
        // Validate input
        if (!currentPassword || !newPassword) {
            return res.status(400).json({ 
                success: false, 
                message: 'Current password and new password are required' 
            });
        }

        if (newPassword.length < 6) {
            return res.status(400).json({ 
                success: false, 
                message: 'New password must be at least 6 characters long' 
            });
        }

        // Get current user
        const user = await dbGet('SELECT * FROM users WHERE id = ?', [managerId]);
        
        if (!user) {
            return res.status(404).json({ success: false, message: 'User not found' });
        }

        // Verify current password
        const isMatch = await bcrypt.compare(currentPassword, user.password);
        if (!isMatch) {
            return res.status(401).json({ 
                success: false, 
                message: 'Current password is incorrect' 
            });
        }

        // Hash new password and update
        const hashedPassword = await bcrypt.hash(newPassword, 10);
        await dbRun('UPDATE users SET password = ? WHERE id = ?', [hashedPassword, managerId]);

        res.json({ success: true, message: 'Password changed successfully' });
    } catch (error) {
        console.error('Error changing password:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get notifications
const getNotifications = async (req, res) => {
    const managerId = req.user.id;

    try {
        const pendingCount = await dbGet(
            "SELECT COUNT(*) as count FROM loans WHERE status = 'pending'"
        );

        const overdueLoans = await dbGet(
            `SELECT COUNT(*) as count FROM loans 
            WHERE status IN ('disbursed', 'active') AND next_payment_date < date('now')`
        );

        const notifications = await dbAll(
            `SELECT * FROM notifications 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT 20`,
            [managerId]
        );

        res.json({
            success: true,
            data: {
                pendingApprovals: pendingCount.count,
                overdueLoans: overdueLoans.count,
                notifications: notifications
            }
        });
    } catch (error) {
        console.error('Error fetching notifications:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get all officers (for assignment dropdown)
const getOfficers = async (req, res) => {
    try {
        const officers = await dbAll(
            `SELECT id, name, email, phone FROM users WHERE role = 'officer' ORDER BY name`
        );
        
        res.json({ success: true, officers });
    } catch (error) {
        console.error('Error fetching officers:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get petty cash records
const getPettyCash = async (req, res) => {
    try {
        const pettyCash = await dbAll(
            `SELECT pc.*, u.name as recorded_by_name
            FROM petty_cash pc
            LEFT JOIN users u ON pc.recorded_by = u.id
            ORDER BY pc.transaction_date DESC
            LIMIT 100`
        );
        
        res.json({ success: true, data: pettyCash });
    } catch (error) {
        console.error('Error fetching petty cash:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Add petty cash
const addPettyCash = async (req, res) => {
    const { amount, description } = req.body;
    const managerId = req.user.id;
    
    try {
        // Manager can only ALLOCATE money to petty cash, not record expenses
        // Get current petty cash balance or create if it doesn't exist
        let balance = await dbGet('SELECT * FROM petty_cash_balance ORDER BY id DESC LIMIT 1');
        
        if (!balance) {
            // Create initial balance record (without updated_by to avoid foreign key issues)
            await dbRun(
                `INSERT INTO petty_cash_balance (allocated_amount, current_balance, last_updated)
                VALUES (?, ?, datetime('now'))`,
                [amount, amount]
            );
        } else {
            // Update existing balance - add to both allocated and current balance
            const newAllocated = (balance.allocated_amount || 0) + parseFloat(amount);
            const newCurrent = (balance.current_balance || 0) + parseFloat(amount);
            
            await dbRun(
                `UPDATE petty_cash_balance 
                SET allocated_amount = ?, current_balance = ?, last_updated = datetime('now')
                WHERE id = ?`,
                [newAllocated, newCurrent, balance.id]
            );
        }
        
        // Record the allocation in petty_cash table (positive amount for allocation)
        await dbRun(
            `INSERT INTO petty_cash (amount, description, recorded_by, transaction_date)
            VALUES (?, ?, ?, datetime('now'))`,
            [amount, `ALLOCATION: ${description}`, managerId]
        );
        
        res.json({ 
            success: true, 
            message: 'Petty cash allocated successfully'
        });
    } catch (error) {
        console.error('Error allocating petty cash:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Delete user with validation
const deleteUser = async (req, res) => {
    try {
        const { userId } = req.params;
        
        // Check if user exists
        const user = await dbGet('SELECT * FROM users WHERE id = ?', [userId]);
        if (!user) {
            return res.status(404).json({ success: false, message: 'User not found' });
        }
        
        // Prevent deleting users with active loans
        const activeLoans = await dbGet(
            'SELECT COUNT(*) as count FROM loans WHERE user_id = ? AND status IN (?, ?, ?)',
            [userId, 'pending', 'approved', 'active']
        );
        
        if (activeLoans.count > 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'Cannot delete user with active or pending loans. Please resolve all loans first.' 
            });
        }
        
        // For officers, check if they have assigned clients
        if (user.role === 'officer') {
            const assignedClients = await dbGet(
                'SELECT COUNT(*) as count FROM users WHERE officer_id = ?',
                [userId]
            );
            
            if (assignedClients.count > 0) {
                return res.status(400).json({ 
                    success: false, 
                    message: 'Cannot delete officer with assigned clients. Please reassign clients first.' 
                });
            }
        }
        
        // Delete user
        await dbRun('DELETE FROM users WHERE id = ?', [userId]);
        
        res.json({ success: true, message: 'User deleted successfully' });
    } catch (error) {
        console.error('Error deleting user:', error);
        res.status(500).json({ success: false, message: 'Failed to delete user' });
    }
};

// Clear all notifications
const clearAllNotifications = async (req, res) => {
    try {
        const userId = req.user.userId;
        await dbRun('DELETE FROM notifications WHERE user_id = ?', [userId]);
        res.json({ success: true, message: 'All notifications cleared' });
    } catch (error) {
        console.error('Error clearing notifications:', error);
        res.status(500).json({ success: false, message: 'Failed to clear notifications' });
    }
};

// Get single user
const getUser = async (req, res) => {
    try {
        const { userId } = req.params;
        const user = await dbGet('SELECT id, name, email, phone, role, status, created_at, officer_id FROM users WHERE id = ?', [userId]);
        
        if (!user) {
            return res.status(404).json({ success: false, message: 'User not found' });
        }
        
        res.json({ success: true, data: user });
    } catch (error) {
        console.error('Error getting user:', error);
        res.status(500).json({ success: false, message: 'Failed to get user details' });
    }
};

// Update user
const updateUser = async (req, res) => {
    try {
        const { userId } = req.params;
        const { name, email, phone, role, status } = req.body;
        
        // Check if user exists
        const user = await dbGet('SELECT * FROM users WHERE id = ?', [userId]);
        if (!user) {
            return res.status(404).json({ success: false, message: 'User not found' });
        }
        
        // Check if email is already taken by another user
        const existingUser = await dbGet('SELECT * FROM users WHERE email = ? AND id != ?', [email, userId]);
        if (existingUser) {
            return res.status(400).json({ success: false, message: 'Email already in use' });
        }
        
        // Update user
        await dbRun(
            'UPDATE users SET name = ?, email = ?, phone = ?, role = ?, status = ? WHERE id = ?',
            [name, email, phone, role, status, userId]
        );
        
        res.json({ success: true, message: 'User updated successfully' });
    } catch (error) {
        console.error('Error updating user:', error);
        res.status(500).json({ success: false, message: 'Failed to update user' });
    }
};

// Assign client to officer
const assignClientToOfficer = async (req, res) => {
    try {
        const { clientId } = req.params;
        const { officer_id } = req.body;
        
        // Check if client exists
        const client = await dbGet('SELECT * FROM users WHERE id = ? AND role = ?', [clientId, 'client']);
        if (!client) {
            return res.status(404).json({ success: false, message: 'Client not found' });
        }
        
        // Check if officer exists
        const officer = await dbGet('SELECT * FROM users WHERE id = ? AND role = ?', [officer_id, 'officer']);
        if (!officer) {
            return res.status(404).json({ success: false, message: 'Officer not found' });
        }
        
        // Update client's officer assignment
        await dbRun('UPDATE users SET officer_id = ? WHERE id = ?', [officer_id, clientId]);
        
        res.json({ success: true, message: 'Client reassigned successfully' });
    } catch (error) {
        console.error('Error assigning client:', error);
        res.status(500).json({ success: false, message: 'Failed to reassign client' });
    }
};

module.exports = {
    getDashboardStats,
    getPendingLoans,
    updateLoanStatus,
    getLoanSettings,
    updateLoanSettings,
    getAllUsers,
    getUser,
    createUser,
    updateUser,
    resetPassword,
    updateUserStatus,
    changePassword,
    getNotifications,
    getOfficers,
    getPettyCash,
    addPettyCash,
    getFinancialOversight,
    getSystemSettings,
    updateSystemSettings,
    deleteUser,
    clearAllNotifications,
    assignClientToOfficer
};