const { dbRun, dbGet, dbAll } = require('../db');

// Create loan application with full details
exports.createLoanApplication = async (req, res) => {
    try {
        const {
            client_id, loan_type, amount, purpose, duration_months,
            repayment_mode, payment_day, interest_rate, processing_fee,
            guarantors, lc1_name, lc1_phone,
            collateral_type, collateral_description, collateral_value
        } = req.body;

        const officer_id = req.user.id;
        const form_fee = 5000;

        // Calculate totals using simple flat interest rate
        // Balance = Amount × (1 + Interest Rate / 100)
        // Example: 100,000 at 20% = 100,000 × 1.20 = 120,000
        // NOTE: Processing fee is deducted at disbursement, NOT added to balance
        // NOTE: Application fee is paid separately before loan application
        const interest_amount = (amount * interest_rate) / 100;
        const total_repayable = parseFloat(amount) + parseFloat(interest_amount);
        
        // Calculate installment based on repayment mode
        let numberOfPayments;
        
        switch(repayment_mode) {
            case 'daily':
                numberOfPayments = duration_months * 30; // 30 days per month
                break;
            case 'weekly':
                numberOfPayments = duration_months * 4; // 4 weeks per month
                break;
            case 'bi-weekly':
            case 'biweekly':
                numberOfPayments = duration_months * 2; // 2 bi-weekly periods per month
                break;
            case 'monthly':
            default:
                numberOfPayments = duration_months; // 1 payment per month
                break;
        }
        
        const installment_amount = total_repayable / numberOfPayments;

        // Handle file uploads
        const lc1_letter_url = req.files && req.files.lc1_letter 
            ? `/uploads/${req.files.lc1_letter[0].filename}` 
            : null;
        
        const collateral_photo_url = req.files && req.files.collateral_photo 
            ? `/uploads/${req.files.collateral_photo[0].filename}` 
            : null;

        // Insert loan
        const loanResult = await dbRun(`
            INSERT INTO loans (
                user_id, officer_id, loan_type, amount, interest_rate,
                purpose, duration_months, repayment_mode, payment_day,
                processing_fee, form_fee, balance, installment_amount,
                lc1_letter_url, lc1_name, lc1_phone,
                collateral_type, collateral_description, collateral_value, collateral_photo_url,
                status, agreement_date, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', date('now'), datetime('now'))
        `, [
            client_id, officer_id, loan_type, amount, interest_rate,
            purpose, duration_months, repayment_mode, payment_day,
            processing_fee, form_fee, total_repayable, installment_amount,
            lc1_letter_url, lc1_name, lc1_phone,
            collateral_type, collateral_description, collateral_value, collateral_photo_url
        ]);

        const loanId = loanResult.lastID;

        // Insert guarantors
        if (guarantors && Array.isArray(guarantors)) {
            for (const guarantor of guarantors) {
                await dbRun(`
                    INSERT INTO guarantors (
                        loan_id, name, phone, national_id, relationship,
                        village, photo_url, id_photo_front, id_photo_back
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                `, [
                    loanId, guarantor.name, guarantor.phone, guarantor.national_id,
                    guarantor.relationship, guarantor.village,
                    guarantor.photo_url, guarantor.id_photo_front, guarantor.id_photo_back
                ]);
            }
        }

        res.status(201).json({
            success: true,
            message: 'Loan application created successfully',
            loanId
        });
    } catch (error) {
        console.error('Loan application error:', error);
        res.status(500).json({ error: 'Failed to create loan application' });
    }
};

// Get loan agreement details for printing
exports.getLoanAgreement = async (req, res) => {
    try {
        const { loanId } = req.params;

        const loan = await dbGet(`
            SELECT 
                l.*,
                u.name as client_name, u.phone as client_phone, u.email as client_email,
                u.village, u.parish, u.division, u.occupation,
                o.name as officer_name, o.phone as officer_phone,
                ls.interest_rate as settings_interest_rate, ls.penalty_rate, ls.early_payment_bonus
            FROM loans l
            JOIN users u ON l.user_id = u.id
            LEFT JOIN users o ON l.officer_id = o.id
            LEFT JOIN loan_settings ls ON l.loan_type = ls.loan_type
            WHERE l.id = ?
        `, [loanId]);

        if (!loan) {
            return res.status(404).json({ error: 'Loan not found' });
        }

        const guarantors = await dbAll(`
            SELECT * FROM guarantors WHERE loan_id = ?
        `, [loanId]);

        // Calculate amounts using flat interest rate
        // Balance = Amount × (1 + Interest Rate / 100)
        // Processing fee and application fee are NOT added to balance
        const interest_amount = (loan.amount * loan.interest_rate) / 100;
        const total_repayable = parseFloat(loan.amount) + parseFloat(interest_amount);

        res.json({
            success: true,
            loan: {
                ...loan,
                interest_amount,
                total_repayable,
                guarantors
            }
        });
    } catch (error) {
        console.error('Error fetching loan agreement:', error);
        res.status(500).json({ error: 'Failed to fetch loan agreement' });
    }
};

// Upload guarantor photos
exports.uploadGuarantorPhotos = async (req, res) => {
    try {
        const { guarantorId } = req.params;
        
        const photo_url = req.files?.photo ? `/uploads/${req.files.photo[0].filename}` : null;
        const id_photo_front = req.files?.id_front ? `/uploads/${req.files.id_front[0].filename}` : null;
        const id_photo_back = req.files?.id_back ? `/uploads/${req.files.id_back[0].filename}` : null;

        await dbRun(`
            UPDATE guarantors 
            SET photo_url = COALESCE(?, photo_url),
                id_photo_front = COALESCE(?, id_photo_front),
                id_photo_back = COALESCE(?, id_photo_back)
            WHERE id = ?
        `, [photo_url, id_photo_front, id_photo_back, guarantorId]);

        res.json({ success: true, message: 'Photos uploaded successfully' });
    } catch (error) {
        console.error('Photo upload error:', error);
        res.status(500).json({ error: 'Failed to upload photos' });
    }
};

// Upload/Update collateral information for a loan
exports.uploadCollateral = async (req, res) => {
    try {
        const { loanId } = req.params;
        const { collateral_type, collateral_description, collateral_value } = req.body;
        
        // Handle collateral photo upload
        const collateral_photo_url = req.file ? `/uploads/${req.file.filename}` : null;

        // Build dynamic update query
        const updates = [];
        const params = [];

        if (collateral_type) {
            updates.push('collateral_type = ?');
            params.push(collateral_type);
        }
        if (collateral_description) {
            updates.push('collateral_description = ?');
            params.push(collateral_description);
        }
        if (collateral_value) {
            updates.push('collateral_value = ?');
            params.push(collateral_value);
        }
        if (collateral_photo_url) {
            updates.push('collateral_photo_url = ?');
            params.push(collateral_photo_url);
        }

        if (updates.length === 0) {
            return res.status(400).json({ error: 'No collateral information provided' });
        }

        params.push(loanId);

        await dbRun(`
            UPDATE loans 
            SET ${updates.join(', ')}
            WHERE id = ?
        `, params);

        res.json({ 
            success: true, 
            message: 'Collateral information updated successfully',
            collateral_photo_url
        });
    } catch (error) {
        console.error('Collateral upload error:', error);
        res.status(500).json({ error: 'Failed to upload collateral information' });
    }
};

module.exports = exports;
