const { dbGet, dbAll } = require('../db');

// Generate Income Statement (Profit & Loss)
exports.generateIncomeStatement = async (req, res) => {
    try {
        const { startDate, endDate } = req.query;
        
        // Revenue - Use transactions for repayments
        const repayments = await dbGet(`
            SELECT COALESCE(SUM(amount), 0) as total
            FROM transactions
            WHERE transaction_type = 'repayment' AND transaction_date BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        // Calculate interest as percentage of loans
        const loanInterest = await dbGet(`
            SELECT COALESCE(SUM(amount * interest_rate / 100), 0) as total
            FROM loans
            WHERE created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        const processingFees = await dbGet(`
            SELECT COALESCE(SUM(processing_fee + form_fee), 0) as total
            FROM loans
            WHERE created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        // Expenses
        const operatingExpenses = await dbGet(`
            SELECT COALESCE(SUM(amount), 0) as total
            FROM transactions
            WHERE transaction_type = 'expenditure'
            AND created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        const pettyCashExpenses = await dbGet(`
            SELECT COALESCE(SUM(amount), 0) as total
            FROM petty_cash
            WHERE created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        const totalRevenue = (loanInterest.total || 0) + (processingFees.total || 0) + (penaltyFees.total || 0);
        const totalExpenses = (operatingExpenses.total || 0) + (pettyCashExpenses.total || 0);
        const netIncome = totalRevenue - totalExpenses;
        
        res.json({
            success: true,
            data: {
                revenue: {
                    loanInterest: loanInterest.total || 0,
                    processingFees: processingFees.total || 0,
                    penaltyFees: penaltyFees.total || 0,
                    total: totalRevenue
                },
                expenses: {
                    operating: operatingExpenses.total || 0,
                    pettyCash: pettyCashExpenses.total || 0,
                    total: totalExpenses
                },
                netIncome: netIncome,
                profitMargin: totalRevenue > 0 ? ((netIncome / totalRevenue) * 100).toFixed(2) : 0
            }
        });
    } catch (error) {
        console.error('Income statement error:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Generate Balance Sheet
exports.generateBalanceSheet = async (req, res) => {
    try {
        const { asOfDate } = req.query;
        
        // Assets
        const cashOnHand = await dbGet(`
            SELECT 
                COALESCE(SUM(CASE WHEN transaction_type IN ('deposit', 'repayment') THEN amount ELSE 0 END), 0) -
                COALESCE(SUM(CASE WHEN transaction_type IN ('disbursement', 'expenditure') THEN amount ELSE 0 END), 0) as total
            FROM transactions
            WHERE created_at <= ?
        `, [asOfDate]);
        
        const loansReceivable = await dbGet(`
            SELECT COALESCE(SUM(balance), 0) as total
            FROM loans
            WHERE status = 'active' AND created_at <= ?
        `, [asOfDate]);
        
        const interestReceivable = await dbGet(`
            SELECT COALESCE(SUM((amount * interest_rate * duration_months / 1200) - 
                   COALESCE((SELECT SUM(amount - principal_amount) FROM repayments WHERE loan_id = loans.id), 0)), 0) as total
            FROM loans
            WHERE status = 'active' AND created_at <= ?
        `, [asOfDate]);
        
        // Liabilities
        const savingsDeposits = await dbGet(`
            SELECT COALESCE(SUM(amount), 0) as total
            FROM transactions
            WHERE transaction_type = 'deposit' AND created_at <= ?
        `, [asOfDate]);
        
        // Equity
        const retainedEarnings = await dbGet(`
            SELECT 
                COALESCE(SUM(CASE WHEN transaction_type IN ('deposit', 'repayment') THEN amount ELSE 0 END), 0) -
                COALESCE(SUM(CASE WHEN transaction_type IN ('disbursement', 'expenditure') THEN amount ELSE 0 END), 0) as total
            FROM transactions
            WHERE created_at <= ?
        `, [asOfDate]);
        
        const totalAssets = (cashOnHand.total || 0) + (loansReceivable.total || 0) + (interestReceivable.total || 0);
        const totalLiabilities = savingsDeposits.total || 0;
        const totalEquity = (retainedEarnings.total || 0) - totalLiabilities;
        
        res.json({
            success: true,
            data: {
                assets: {
                    cash: cashOnHand.total || 0,
                    loansReceivable: loansReceivable.total || 0,
                    interestReceivable: interestReceivable.total || 0,
                    total: totalAssets
                },
                liabilities: {
                    savingsDeposits: savingsDeposits.total || 0,
                    total: totalLiabilities
                },
                equity: {
                    retainedEarnings: totalEquity,
                    total: totalEquity
                },
                totalLiabilitiesAndEquity: totalLiabilities + totalEquity
            }
        });
    } catch (error) {
        console.error('Balance sheet error:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Generate Cash Flow Statement
exports.generateCashFlowStatement = async (req, res) => {
    try {
        const { startDate, endDate } = req.query;
        
        // Operating Activities
        const loanRepayments = await dbGet(`
            SELECT COALESCE(SUM(amount), 0) as total
            FROM repayments
            WHERE payment_date BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        const operatingExpenses = await dbGet(`
            SELECT COALESCE(SUM(amount), 0) as total
            FROM transactions
            WHERE transaction_type = 'expenditure' AND created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        // Financing Activities
        const deposits = await dbGet(`
            SELECT COALESCE(SUM(amount), 0) as total
            FROM transactions
            WHERE transaction_type = 'deposit' AND created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        // Investing Activities
        const loanDisbursements = await dbGet(`
            SELECT COALESCE(SUM(amount), 0) as total
            FROM loans
            WHERE status IN ('active', 'pending', 'approved') AND created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        const netOperating = (loanRepayments.total || 0) - (operatingExpenses.total || 0);
        const netFinancing = deposits.total || 0;
        const netInvesting = -(loanDisbursements.total || 0);
        const netCashFlow = netOperating + netFinancing + netInvesting;
        
        res.json({
            success: true,
            data: {
                operating: {
                    inflows: loanRepayments.total || 0,
                    outflows: operatingExpenses.total || 0,
                    net: netOperating
                },
                financing: {
                    inflows: deposits.total || 0,
                    outflows: 0,
                    net: netFinancing
                },
                investing: {
                    inflows: 0,
                    outflows: loanDisbursements.total || 0,
                    net: netInvesting
                },
                netCashFlow: netCashFlow
            }
        });
    } catch (error) {
        console.error('Cash flow error:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Generate Loan Portfolio Report
exports.generateLoanPortfolioReport = async (req, res) => {
    try {
        const { startDate, endDate } = req.query;
        
        const portfolioStats = await dbGet(`
            SELECT 
                COUNT(*) as total_loans,
                COALESCE(SUM(amount), 0) as total_disbursed,
                COALESCE(SUM(balance), 0) as total_outstanding,
                COALESCE(SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END), 0) as active_loans,
                COALESCE(SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END), 0) as completed_loans,
                COALESCE(SUM(CASE WHEN status = 'defaulted' THEN 1 ELSE 0 END), 0) as defaulted_loans
            FROM loans
            WHERE created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        const loansByType = await dbAll(`
            SELECT 
                loan_type,
                COUNT(*) as count,
                COALESCE(SUM(amount), 0) as total_amount,
                COALESCE(SUM(balance), 0) as outstanding
            FROM loans
            WHERE created_at BETWEEN ? AND ?
            GROUP BY loan_type
        `, [startDate, endDate]);
        
        const repaymentRate = portfolioStats.total_disbursed > 0 
            ? (((portfolioStats.total_disbursed - portfolioStats.total_outstanding) / portfolioStats.total_disbursed) * 100).toFixed(2)
            : 0;
        
        res.json({
            success: true,
            data: {
                summary: portfolioStats,
                repaymentRate: repaymentRate,
                loansByType: loansByType
            }
        });
    } catch (error) {
        console.error('Loan portfolio error:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Generate Collections Report
exports.generateCollectionsReport = async (req, res) => {
    try {
        const { startDate, endDate } = req.query;
        
        const collections = await dbAll(`
            SELECT 
                lr.payment_date,
                u.name as client_name,
                l.loan_type,
                lr.amount,
                lr.payment_method,
                l.interest_rate,
                l.amount as loan_amount
            FROM repayments lr
            JOIN loans l ON lr.loan_id = l.id
            JOIN users u ON l.user_id = u.id
            WHERE lr.payment_date BETWEEN ? AND ?
            ORDER BY lr.payment_date DESC
        `, [startDate, endDate]);
        
        const summary = await dbGet(`
            SELECT 
                COUNT(*) as total_transactions,
                COALESCE(SUM(amount), 0) as total_collected
            FROM repayments
            WHERE payment_date BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        res.json({
            success: true,
            data: {
                summary: summary,
                collections: collections
            }
        });
    } catch (error) {
        console.error('Collections report error:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Generate Disbursements Report
exports.generateDisbursementsReport = async (req, res) => {
    try {
        const { startDate, endDate } = req.query;
        
        const disbursements = await dbAll(`
            SELECT 
                l.created_at as disbursement_date,
                u.name as client_name,
                l.loan_type,
                l.amount,
                l.interest_rate,
                l.duration_months,
                l.processing_fee,
                l.form_fee,
                o.name as officer_name
            FROM loans l
            JOIN users u ON l.user_id = u.id
            LEFT JOIN users o ON l.officer_id = o.id
            WHERE l.created_at BETWEEN ? AND ?
            AND l.status IN ('active', 'completed')
            ORDER BY l.created_at DESC
        `, [startDate, endDate]);
        
        const summary = await dbGet(`
            SELECT 
                COUNT(*) as total_loans,
                COALESCE(SUM(amount), 0) as total_disbursed,
                COALESCE(SUM(processing_fee), 0) as total_processing_fees,
                COALESCE(SUM(form_fee), 0) as total_form_fees
            FROM loans
            WHERE created_at BETWEEN ? AND ?
            AND status IN ('active', 'completed')
        `, [startDate, endDate]);
        
        res.json({
            success: true,
            data: {
                summary: summary,
                disbursements: disbursements
            }
        });
    } catch (error) {
        console.error('Disbursements report error:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Generate Performance Summary
exports.generatePerformanceSummary = async (req, res) => {
    try {
        const { startDate, endDate } = req.query;
        
        // Key Performance Indicators
        const totalClients = await dbGet(`
            SELECT COUNT(DISTINCT user_id) as total
            FROM loans
            WHERE created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        const portfolioGrowth = await dbGet(`
            SELECT 
                COALESCE(SUM(amount), 0) as total_disbursed,
                COALESCE(SUM(balance), 0) as total_outstanding
            FROM loans
            WHERE created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        const repaymentPerformance = await dbGet(`
            SELECT 
                COALESCE(SUM(amount), 0) as total_collected,
                COUNT(*) as total_payments
            FROM repayments
            WHERE payment_date BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        const defaultRate = await dbGet(`
            SELECT 
                COUNT(CASE WHEN status = 'defaulted' THEN 1 END) as defaulted,
                COUNT(*) as total
            FROM loans
            WHERE created_at BETWEEN ? AND ?
        `, [startDate, endDate]);
        
        const officerPerformance = await dbAll(`
            SELECT 
                o.name as officer_name,
                COUNT(l.id) as loans_managed,
                COALESCE(SUM(l.amount), 0) as total_disbursed,
                COALESCE(SUM(lr.amount), 0) as total_collected
            FROM users o
            LEFT JOIN loans l ON l.officer_id = o.id AND l.created_at BETWEEN ? AND ?
            LEFT JOIN repayments lr ON lr.loan_id = l.id
            WHERE o.role = 'officer'
            GROUP BY o.id, o.name
        `, [startDate, endDate]);
        
        res.json({
            success: true,
            data: {
                totalClients: totalClients.total || 0,
                portfolioGrowth: portfolioGrowth,
                repaymentPerformance: repaymentPerformance,
                defaultRate: defaultRate.total > 0 ? ((defaultRate.defaulted / defaultRate.total) * 100).toFixed(2) : 0,
                officerPerformance: officerPerformance
            }
        });
    } catch (error) {
        console.error('Performance summary error:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

module.exports = {
    generateIncomeStatement: exports.generateIncomeStatement,
    generateBalanceSheet: exports.generateBalanceSheet,
    generateCashFlowStatement: exports.generateCashFlowStatement,
    generateLoanPortfolioReport: exports.generateLoanPortfolioReport,
    generateCollectionsReport: exports.generateCollectionsReport,
    generateDisbursementsReport: exports.generateDisbursementsReport,
    generatePerformanceSummary: exports.generatePerformanceSummary
};
