const { dbRun, dbGet, dbAll } = require('../db');
const bcrypt = require('bcryptjs');

// Enroll a new client (Officer or Manager)
exports.enrollClient = async (req, res) => {
    try {
        const {
            name, email, phone, national_id, password,
            village, parish, division, occupation,
            guarantor1_name, guarantor1_phone, guarantor1_email, guarantor1_relationship, guarantor1_address, guarantor1_national_id,
            guarantor2_name, guarantor2_phone, guarantor2_email, guarantor2_relationship, guarantor2_address, guarantor2_national_id
        } = req.body;

        const enrolledBy = req.user.id;
        const officer_id = req.user.role === 'officer' ? req.user.id : req.body.officer_id;

        // Validate guarantors
        if (!guarantor1_name || !guarantor1_phone || !guarantor1_relationship) {
            return res.status(400).json({ 
                success: false,
                error: 'Guarantor 1 information is required (Name, Phone, Relationship)' 
            });
        }
        
        if (!guarantor2_name || !guarantor2_phone || !guarantor2_relationship) {
            return res.status(400).json({ 
                success: false,
                error: 'Guarantor 2 information is required (Name, Phone, Relationship)' 
            });
        }

        // Check if email already exists
        const existing = await dbGet('SELECT id FROM users WHERE email = ?', [email]);
        if (existing) {
            return res.status(400).json({ 
                success: false,
                error: 'Email already registered' 
            });
        }

        // Hash password
        const hashedPassword = await bcrypt.hash(password || 'password123', 10);

        // Handle photo uploads
        const photo_url = req.files?.photo ? `/uploads/${req.files.photo[0].filename}` : null;
        const id_photo_front = req.files?.id_front ? `/uploads/${req.files.id_front[0].filename}` : null;
        const id_photo_back = req.files?.id_back ? `/uploads/${req.files.id_back[0].filename}` : null;
        const lc1_letter_url = req.files?.lc1_letter ? `/uploads/${req.files.lc1_letter[0].filename}` : null;

        // Validate LC1 letter is uploaded
        if (!lc1_letter_url) {
            return res.status(400).json({ 
                success: false,
                error: 'LC1 recommendation letter is required for client enrollment' 
            });
        }

        // Insert client
        const result = await dbRun(`
            INSERT INTO users (
                name, email, phone, national_id, password, role, officer_id,
                photo_url, id_photo_front, id_photo_back, lc1_letter_url,
                village, parish, division, occupation, status
            ) VALUES (?, ?, ?, ?, ?, 'client', ?, ?, ?, ?, ?, ?, ?, ?, ?, 'active')
        `, [
            name, email, phone, national_id, hashedPassword, officer_id,
            photo_url, id_photo_front, id_photo_back, lc1_letter_url,
            village, parish, division, occupation
        ]);

        const clientId = result.lastID;

        // Insert guarantors
        await dbRun(`
            INSERT INTO guarantors (client_id, name, phone, email, relationship, address, national_id)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        `, [clientId, guarantor1_name, guarantor1_phone, guarantor1_email, guarantor1_relationship, guarantor1_address, guarantor1_national_id]);

        await dbRun(`
            INSERT INTO guarantors (client_id, name, phone, email, relationship, address, national_id)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        `, [clientId, guarantor2_name, guarantor2_phone, guarantor2_email, guarantor2_relationship, guarantor2_address, guarantor2_national_id]);

        // Notify officer of new assignment
        if (officer_id) {
            await dbRun(
                `INSERT INTO notifications (user_id, message, type, created_at) VALUES (?, ?, 'alert', datetime('now'))`,
                [officer_id, `New client ${name} has been assigned to you.`]
            );
        }

        res.status(201).json({
            success: true,
            message: `✅ Client enrolled successfully with 2 guarantors${officer_id ? ' and assigned to officer' : ''}`,
            clientId: clientId
        });
    } catch (error) {
        console.error('Enrollment error:', error);
        res.status(500).json({ 
            success: false,
            error: 'Failed to enroll client: ' + error.message 
        });
    }
};

// Get all clients
exports.getAllClients = async (req, res) => {
    try {
        const clients = await dbAll(`
            SELECT 
                u.id, u.name, u.email, u.phone, u.national_id,
                u.village, u.parish, u.division, u.occupation,
                u.photo_url, u.officer_id, u.status, u.created_at,
                o.name as officer_name
            FROM users u
            LEFT JOIN users o ON u.officer_id = o.id
            WHERE u.role = 'client'
            ORDER BY u.created_at DESC
        `);

        res.json({ success: true, clients });
    } catch (error) {
        console.error('Error fetching clients:', error);
        res.status(500).json({ 
            success: false,
            error: 'Failed to fetch clients' 
        });
    }
};

// Assign/Reassign client to officer (Manager only)
exports.assignClientToOfficer = async (req, res) => {
    try {
        const { clientId, officerId } = req.body;

        await dbRun('UPDATE users SET officer_id = ? WHERE id = ? AND role = ?', 
            [officerId, clientId, 'client']);

        res.json({ success: true, message: 'Client assigned successfully' });
    } catch (error) {
        console.error('Assignment error:', error);
        res.status(500).json({ 
            success: false,
            error: 'Failed to assign client' 
        });
    }
};

// Reset user password (Manager only)
exports.resetUserPassword = async (req, res) => {
    try {
        const { userId, newPassword } = req.body;

        const hashedPassword = await bcrypt.hash(newPassword, 10);
        
        await dbRun('UPDATE users SET password = ? WHERE id = ?', [hashedPassword, userId]);

        res.json({ success: true, message: 'Password reset successfully' });
    } catch (error) {
        console.error('Password reset error:', error);
        res.status(500).json({ 
            success: false,
            error: 'Failed to reset password' 
        });
    }
};

// Get client details with loans and guarantors
exports.getClientDetails = async (req, res) => {
    try {
        const { clientId } = req.params;

        const client = await dbGet(`
            SELECT 
                u.*, o.name as officer_name
            FROM users u
            LEFT JOIN users o ON u.officer_id = o.id
            WHERE u.id = ?
        `, [clientId]);

        if (!client) {
            return res.status(404).json({ 
                success: false,
                error: 'Client not found' 
            });
        }

        // Get guarantors for this client
        const guarantors = await dbAll(`
            SELECT id, name, phone, email, relationship, address, national_id, created_at
            FROM guarantors
            WHERE client_id = ?
            ORDER BY id
        `, [clientId]);

        res.json({ success: true, client, guarantors });
    } catch (error) {
        console.error('Error fetching client details:', error);
        res.status(500).json({ 
            success: false,
            error: 'Failed to fetch client details' 
        });
    }
};

// Update client details
exports.updateClientDetails = async (req, res) => {
    try {
        const { clientId } = req.params;
        const { name, email, phone, national_id, village, parish, division, occupation, status } = req.body;

        // Check if client exists
        const client = await dbGet('SELECT id FROM users WHERE id = ? AND role = "client"', [clientId]);
        if (!client) {
            return res.status(404).json({ 
                success: false,
                error: 'Client not found' 
            });
        }

        // Check if email is being changed and if it's already taken by another user
        if (email) {
            const existingEmail = await dbGet('SELECT id FROM users WHERE email = ? AND id != ?', [email, clientId]);
            if (existingEmail) {
                return res.status(400).json({ 
                    success: false,
                    error: 'Email already in use by another user' 
                });
            }
        }

        // Update client information
        await dbRun(`
            UPDATE users 
            SET name = ?, email = ?, phone = ?, national_id = ?, 
                village = ?, parish = ?, division = ?, occupation = ?, status = ?
            WHERE id = ?
        `, [name, email, phone, national_id, village, parish, division, occupation, status, clientId]);

        res.json({ 
            success: true, 
            message: 'Client details updated successfully' 
        });
    } catch (error) {
        console.error('Error updating client details:', error);
        res.status(500).json({ 
            success: false,
            error: 'Failed to update client details' 
        });
    }
};

// Get client loans
exports.getClientLoans = async (req, res) => {
    try {
        const { clientId } = req.params;

        const loans = await dbAll(`
            SELECT * FROM loans WHERE user_id = ? ORDER BY created_at DESC
        `, [clientId]);

        res.json({ success: true, loans });
    } catch (error) {
        console.error('Error fetching client loans:', error);
        res.status(500).json({ 
            success: false,
            error: 'Failed to fetch loans' 
        });
    }
};

module.exports = exports;

