const { dbAll, dbGet, dbRun } = require('../db');

// Get loan overview for client
const getLoanOverview = async (req, res) => {
    const clientId = req.user.id;

    try {
        const loan = await dbGet(
            `SELECT * FROM loans WHERE user_id = ? ORDER BY created_at DESC LIMIT 1`,
            [clientId]
        );

        if (!loan) {
            return res.json({ 
                success: true, 
                data: null,
                isNewClient: true,
                eligibleForTopup: false,
                message: 'No active loan' 
            });
        }

        // Check if eligible for top-up
        // Eligible if: loan is disbursed/active/completed AND at least 80% has been paid
        const amountPaid = loan.amount_paid || 0;
        const totalAmount = loan.amount || 0;
        const repaymentPercentage = totalAmount > 0 ? (amountPaid / totalAmount) * 100 : 0;
        
        const isDisbursed = ['disbursed', 'active', 'completed', 'overdue'].includes(loan.status);
        const eligibleForTopup = isDisbursed && repaymentPercentage >= 80;

        res.json({ 
            success: true, 
            data: loan,
            isNewClient: false,
            eligibleForTopup: eligibleForTopup,
            repaymentPercentage: repaymentPercentage.toFixed(2)
        });
    } catch (error) {
        console.error('Error fetching loan overview:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get repayment progress
const getRepaymentProgress = async (req, res) => {
    const clientId = req.user.id;

    try {
        const loan = await dbGet(
            `SELECT 
                l.*,
                CASE 
                    WHEN l.next_payment_date < date('now') THEN 
                        CAST(julianday('now') - julianday(l.next_payment_date) AS INTEGER) * (l.balance * 0.01)
                    ELSE 0
                END as penalty
            FROM loans l
            WHERE l.user_id = ? AND l.status IN ('disbursed', 'active', 'overdue')
            ORDER BY l.created_at DESC 
            LIMIT 1`,
            [clientId]
        );

        if (!loan) {
            return res.json({ 
                success: true, 
                data: null,
                message: 'No active loan' 
            });
        }

        // Get repayment history
        const repayments = await dbAll(
            `SELECT * FROM repayments WHERE loan_id = ? ORDER BY payment_date DESC`,
            [loan.id]
        );

        // Calculate next payment amount based on duration
        const periods = loan.duration_months || 12;
        const nextPaymentAmount = (loan.balance / periods).toFixed(2);

        res.json({ 
            success: true, 
            data: {
                loan: loan,
                repaymentHistory: repayments,
                nextPaymentAmount: nextPaymentAmount,
                progressPercentage: ((loan.amount_paid / loan.amount) * 100).toFixed(2)
            }
        });
    } catch (error) {
        console.error('Error fetching repayment progress:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get portfolio growth
const getPortfolioGrowth = async (req, res) => {
    const clientId = req.user.id;

    try {
        // Get client's current active loan
        const currentLoan = await dbGet(
            `SELECT 
                id,
                disbursed_amount as initial_loan,
                balance,
                amount_paid,
                status
            FROM loans 
            WHERE user_id = ? AND status IN ('disbursed', 'active')
            ORDER BY disbursed_at DESC 
            LIMIT 1`,
            [clientId]
        );

        // Get client's loan history (completed loans)
        const loanHistory = await dbAll(
            `SELECT disbursed_amount FROM loans 
            WHERE user_id = ? AND status = 'completed'
            ORDER BY disbursed_at DESC`,
            [clientId]
        );

        let eligibleLimit = 100000; // Default starting limit
        let growthRate = 0;
        let growthMessage = 'Complete your first loan to unlock growth';
        let qualifiesForTopup = false;
        let nextMilestone = '';
        let progressPercent = 0;

        if (currentLoan) {
            const initialLoan = parseFloat(currentLoan.initial_loan) || 0;
            const amountPaid = parseFloat(currentLoan.amount_paid) || 0;
            const totalOwed = initialLoan + (initialLoan * 0.44 * 3 / 12); // Assuming 44% rate, 3 months
            const repaymentProgress = totalOwed > 0 ? (amountPaid / totalOwed) * 100 : 0;

            // Portfolio Growth Logic:
            // Initial loan: 100K
            // After 50% paid: +10% (110K eligible)
            // After 80% paid: +20% (120K eligible) + qualifies for top-up
            
            if (repaymentProgress >= 80) {
                // 80%+ paid: 30% growth total, qualifies for top-up
                growthRate = 30;
                eligibleLimit = Math.round(initialLoan * 1.30);
                qualifiesForTopup = true;
                growthMessage = '🎉 Excellent! You qualify for a top-up loan!';
                nextMilestone = 'Complete this loan to unlock even higher limits';
                progressPercent = 100;
            } else if (repaymentProgress >= 50) {
                // 50-79% paid: 20% growth, working toward top-up
                growthRate = 20;
                eligibleLimit = Math.round(initialLoan * 1.20);
                qualifiesForTopup = false;
                growthMessage = `Great progress! Pay ${(80 - repaymentProgress).toFixed(0)}% more to unlock top-up`;
                nextMilestone = `${(80 - repaymentProgress).toFixed(0)}% away from top-up eligibility (30% growth)`;
                progressPercent = (repaymentProgress / 80) * 100;
            } else if (repaymentProgress > 0) {
                // 1-49% paid: 10% growth
                growthRate = 10;
                eligibleLimit = Math.round(initialLoan * 1.10);
                qualifiesForTopup = false;
                growthMessage = `Keep going! Pay ${(50 - repaymentProgress).toFixed(0)}% more to unlock 20% growth`;
                nextMilestone = `${(50 - repaymentProgress).toFixed(0)}% away from next milestone (20% growth)`;
                progressPercent = (repaymentProgress / 50) * 100;
            } else {
                // Just started: 10% potential growth
                growthRate = 10;
                eligibleLimit = Math.round(initialLoan * 1.10);
                growthMessage = 'Start making payments to unlock your growth potential';
                nextMilestone = 'Pay 50% to unlock 20% growth rate';
                progressPercent = 0;
            }
        } else if (loanHistory.length > 0) {
            // Client has completed loans but no active loan
            const lastCompletedLoan = parseFloat(loanHistory[0].disbursed_amount) || 100000;
            eligibleLimit = Math.round(lastCompletedLoan * 1.30); // 30% increase after completion
            growthRate = 30;
            growthMessage = '✅ Previous loan completed! Apply for a new loan';
            progressPercent = 100;
        }

        res.json({ 
            success: true, 
            data: {
                eligible_limit: eligibleLimit,
                growth_rate: growthRate,
                growth_message: growthMessage,
                qualifies_for_topup: qualifiesForTopup,
                next_milestone: nextMilestone,
                progress_percent: progressPercent,
                current_loan: currentLoan ? {
                    amount: currentLoan.initial_loan,
                    paid: currentLoan.amount_paid,
                    balance: currentLoan.balance
                } : null
            }
        });
    } catch (error) {
        console.error('Error fetching portfolio growth:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Request loan top-up (Must have paid at least 80% of current loan)
const requestLoanTopUp = async (req, res) => {
    const clientId = req.user.id;
    const { amount, loan_type, repayment_mode } = req.body;

    try {
        // Check if client has an active loan
        const activeLoan = await dbGet(
            `SELECT * FROM loans WHERE user_id = ? AND status IN ('active', 'disbursed') ORDER BY id DESC LIMIT 1`,
            [clientId]
        );

        if (!activeLoan) {
            return res.status(400).json({ 
                success: false, 
                message: 'You need an existing loan to request a top-up. Please apply for a new loan instead.' 
            });
        }

        // Check if client has paid at least 80% of the loan
        const loanAmount = parseFloat(activeLoan.amount);
        const amountPaid = parseFloat(activeLoan.amount_paid);
        const percentagePaid = (amountPaid / loanAmount) * 100;

        if (percentagePaid < 80) {
            return res.status(400).json({ 
                success: false, 
                message: `You must pay at least 80% of your current loan before requesting a top-up. You have paid ${percentagePaid.toFixed(1)}%. Please pay ${(80 - percentagePaid).toFixed(1)}% more.` 
            });
        }

        // Calculate new loan amount: current balance + top-up amount
        const currentBalance = parseFloat(activeLoan.balance);
        const topupAmount = parseFloat(amount);
        const newLoanAmount = currentBalance + topupAmount;

        // Get interest rate from settings (default to 20% if not found)
        const settings = await dbGet(
            'SELECT interest_rate FROM loan_settings WHERE loan_type = ?',
            [loan_type]
        );

        const interestRate = settings ? settings.interest_rate : 20;

        // Mark the old loan as completed/topped-up
        await dbRun(
            `UPDATE loans SET status = 'completed', updated_at = datetime('now') WHERE id = ?`,
            [activeLoan.id]
        );

        // Create new loan application with combined amount (LC1 letter already on file from enrollment)
        await dbRun(
            `INSERT INTO loans (user_id, officer_id, loan_type, amount, interest_rate, balance, repayment_mode, status, duration_months, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', ?, datetime('now'))`,
            [clientId, activeLoan.officer_id, loan_type, newLoanAmount, interestRate, newLoanAmount, repayment_mode, activeLoan.duration_months]
        );

        // Create notification for client
        await dbRun(
            `INSERT INTO notifications (user_id, message, type, created_at) VALUES (?, ?, ?, datetime('now'))`,
            [clientId, `Your loan top-up request for UGX ${topupAmount.toLocaleString()} has been submitted. New loan amount will be UGX ${newLoanAmount.toLocaleString()}.`, 'approval']
        );

        // Notify officer if assigned
        if (activeLoan.officer_id) {
            await dbRun(
                `INSERT INTO notifications (user_id, message, type, created_at) VALUES (?, ?, ?, datetime('now'))`,
                [activeLoan.officer_id, `Client has requested a loan top-up of UGX ${topupAmount.toLocaleString()}. New loan amount: UGX ${newLoanAmount.toLocaleString()}.`, 'alert']
            );
        }

        res.json({ 
            success: true, 
            message: 'Loan top-up request submitted successfully. You have paid ' + percentagePaid.toFixed(1) + '% of your loan.',
            data: {
                previousBalance: currentBalance,
                topupAmount: topupAmount,
                newLoanAmount: newLoanAmount,
                percentagePaid: percentagePaid.toFixed(1)
            }
        });
    } catch (error) {
        console.error('Error requesting loan top-up:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get notifications
const getNotifications = async (req, res) => {
    const clientId = req.user.id;

    try {
        const notifications = await dbAll(
            `SELECT * FROM notifications 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT 20`,
            [clientId]
        );

        res.json({ success: true, data: notifications });
    } catch (error) {
        console.error('Error fetching notifications:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Mark notification as read
const markNotificationRead = async (req, res) => {
    const { notificationId } = req.params;

    try {
        await dbRun(
            'UPDATE notifications SET is_read = 1 WHERE id = ?',
            [notificationId]
        );

        res.json({ success: true, message: 'Notification marked as read' });
    } catch (error) {
        console.error('Error marking notification as read:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Apply for a loan (client initiates)
const applyForLoan = async (req, res) => {
    const clientId = req.user.id;
    
    try {
        const {
            amount, loan_type, purpose, duration_months,
            repayment_mode, collateral_type, collateral_value, collateral_description
        } = req.body;

        // Validate required fields
        if (!amount || !loan_type || !purpose || !duration_months || !repayment_mode) {
            return res.status(400).json({ 
                success: false, 
                message: 'Please fill in all required fields' 
            });
        }

        // STRICT CHECK: Client MUST have paid and accountant-confirmed application fee
        const confirmedFee = await dbGet(
            `SELECT * FROM loan_fees 
            WHERE user_id = ? 
            AND loan_id IS NULL 
            AND fee_type = 'application'
            AND status = 'confirmed'
            AND payment_confirmed_by IS NOT NULL
            ORDER BY created_at DESC LIMIT 1`,
            [clientId]
        );

        if (!confirmedFee) {
            return res.status(403).json({ 
                success: false, 
                message: 'You must pay the application fee and wait for accountant confirmation before applying for a loan.',
                requiresApplicationFee: true
            });
        }

        // Check if this fee was already used for another loan
        const feeAlreadyUsed = await dbGet(
            `SELECT * FROM loans WHERE application_fee_id = ?`,
            [confirmedFee.id]
        );

        if (feeAlreadyUsed) {
            return res.status(403).json({ 
                success: false, 
                message: 'This application fee has already been used. Please pay a new application fee.',
                requiresApplicationFee: true
            });
        }

        // Check if client already has a pending or active loan
        const existingLoan = await dbGet(
            `SELECT * FROM loans 
            WHERE user_id = ? AND status IN ('pending', 'approved', 'disbursed', 'active', 'overdue')
            ORDER BY created_at DESC LIMIT 1`,
            [clientId]
        );

        if (existingLoan) {
            return res.status(400).json({ 
                success: false, 
                message: `You already have a ${existingLoan.status} loan. Please wait until it's completed or consider a top-up if you've paid 80%+.` 
            });
        }

        // Get client info to find assigned officer
        const client = await dbGet(
            'SELECT officer_id, name FROM users WHERE id = ?',
            [clientId]
        );

        if (!client || !client.officer_id) {
            return res.status(400).json({ 
                success: false, 
                message: 'You are not assigned to a credit officer yet. Please contact the manager.' 
            });
        }

        // Get default interest rate from loan settings
        const loanSettings = await dbGet(
            'SELECT interest_rate FROM loan_settings WHERE loan_type = ?',
            [loan_type]
        );
        
        const interest_rate = loanSettings ? loanSettings.interest_rate : 20; // Default 20% FLAT

        // Get system settings including processing fee tiers
        const systemSettings = await dbGet('SELECT * FROM system_settings WHERE id = 1');
        const form_fee = systemSettings ? (systemSettings.application_fee || 5000) : 5000;

        // Calculate processing fee based on loan amount tiers
        let processing_fee = 0;
        if (systemSettings) {
            const tier1Max = systemSettings.processing_fee_tier1_max || 199999;
            const tier2Max = systemSettings.processing_fee_tier2_max || 500000;
            const tier1Amount = systemSettings.processing_fee_tier1_amount || 5000;
            const tier2Amount = systemSettings.processing_fee_tier2_amount || 15000;
            const tier3Amount = systemSettings.processing_fee_tier3_amount || 20000;

            if (amount <= tier1Max) {
                // 50k - 199k = 5k
                processing_fee = tier1Amount;
            } else if (amount <= tier2Max) {
                // 200k - 500k = 15k
                processing_fee = tier2Amount;
            } else {
                // 500k+ = 20k
                processing_fee = tier3Amount;
            }
        }

        // Calculate loan totals using simple flat rate
        // Balance = Amount × (1 + Interest Rate / 100)
        // Example: 100,000 at 20% = 120,000
        // NOTE: Processing fee is deducted at disbursement, NOT added to balance
        // NOTE: Application fee is paid separately before loan application
        const interest_amount = (amount * interest_rate) / 100;
        const total_repayable = parseFloat(amount) + parseFloat(interest_amount);
        
        // Calculate installment based on repayment mode
        let numberOfPayments;
        
        switch(repayment_mode) {
            case 'daily':
                numberOfPayments = duration_months * 30; // 30 days per month
                break;
            case 'weekly':
                numberOfPayments = duration_months * 4; // 4 weeks per month
                break;
            case 'bi-weekly':
            case 'biweekly':
                numberOfPayments = duration_months * 2; // 2 bi-weekly periods per month
                break;
            case 'monthly':
            default:
                numberOfPayments = duration_months; // 1 payment per month
                break;
        }
        
        const installment_amount = total_repayable / numberOfPayments;

        // Create loan application with link to application fee
        const result = await dbRun(`
            INSERT INTO loans (
                user_id, officer_id, loan_type, amount, interest_rate,
                purpose, duration_months, repayment_mode, 
                processing_fee, form_fee, balance, installment_amount,
                collateral_type, collateral_description, collateral_value,
                application_fee_id,
                status, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', datetime('now'))
        `, [
            clientId, client.officer_id, loan_type, amount, interest_rate,
            purpose, duration_months, repayment_mode,
            processing_fee, form_fee, total_repayable, installment_amount,
            collateral_type, collateral_description, collateral_value,
            confirmedFee.id
        ]);

        const loanId = result.lastID;

        // Link the application fee to this loan
        await dbRun(
            `UPDATE loan_fees SET loan_id = ? WHERE id = ?`,
            [loanId, confirmedFee.id]
        );

        // Notify the officer
        await dbRun(
            `INSERT INTO notifications (user_id, message, type, created_at) VALUES (?, ?, 'alert', datetime('now'))`,
            [client.officer_id, `New loan application from ${client.name} for UGX ${Number(amount).toLocaleString()}.`]
        );

        // Notify the client
        await dbRun(
            `INSERT INTO notifications (user_id, message, type, created_at) VALUES (?, ?, 'info', datetime('now'))`,
            [clientId, `Your loan application for UGX ${Number(amount).toLocaleString()} has been submitted successfully.`]
        );

        res.status(201).json({
            success: true,
            message: 'Loan application submitted successfully! Your credit officer will review it soon.',
            loanId: loanId
        });
    } catch (error) {
        console.error('Error submitting loan application:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Clear all notifications
const clearAllNotifications = async (req, res) => {
    try {
        const userId = req.user.userId;
        await dbRun('DELETE FROM notifications WHERE user_id = ?', [userId]);
        res.json({ success: true, message: 'All notifications cleared' });
    } catch (error) {
        console.error('Error clearing notifications:', error);
        res.status(500).json({ success: false, message: 'Failed to clear notifications' });
    }
};

// Pay application fee (Step 1 before loan application)
const payApplicationFee = async (req, res) => {
    const clientId = req.user.id;
    
    try {
        const { amount, payment_method, transaction_reference } = req.body;

        // Get system settings for application fee amount
        const feeSetting = await dbGet(
            'SELECT setting_value FROM system_settings WHERE setting_key = ?',
            ['application_fee_amount']
        );
        const requiredFee = feeSetting ? parseFloat(feeSetting.setting_value) : 5000;

        if (parseFloat(amount) < requiredFee) {
            return res.status(400).json({ 
                success: false, 
                message: `Application fee must be at least UGX ${requiredFee.toLocaleString()}`
            });
        }

        // Check if client already has a pending unconfirmed application fee
        const pendingFee = await dbGet(
            `SELECT * FROM loan_fees 
            WHERE user_id = ? 
            AND loan_id IS NULL 
            AND fee_type = 'application'
            AND status = 'pending'
            ORDER BY created_at DESC LIMIT 1`,
            [clientId]
        );

        if (pendingFee) {
            return res.status(400).json({ 
                success: false, 
                message: 'You already have a pending application fee awaiting accountant confirmation.'
            });
        }

        // Check if client has an unused confirmed fee
        const unusedFee = await dbGet(
            `SELECT * FROM loan_fees 
            WHERE user_id = ? 
            AND loan_id IS NULL 
            AND fee_type = 'application'
            AND status = 'confirmed'
            AND payment_confirmed_by IS NOT NULL
            ORDER BY created_at DESC LIMIT 1`,
            [clientId]
        );

        if (unusedFee) {
            return res.status(400).json({ 
                success: false, 
                message: 'You already have a confirmed application fee. You can proceed to apply for a loan.'
            });
        }

        // Record the application fee payment
        const result = await dbRun(`
            INSERT INTO loan_fees (
                user_id, fee_type, amount, payment_method,
                transaction_reference, status, created_at
            ) VALUES (?, 'application', ?, ?, ?, 'pending', datetime('now'))
        `, [clientId, amount, payment_method, transaction_reference || null]);

        // Get client name
        const client = await dbGet('SELECT name FROM users WHERE id = ?', [clientId]);

        // Notify accountant to confirm payment
        const accountants = await dbAll('SELECT id FROM users WHERE role = "accountant"');
        for (const accountant of accountants) {
            await dbRun(
                `INSERT INTO notifications (user_id, message, type, created_at) 
                VALUES (?, ?, 'alert', datetime('now'))`,
                [accountant.id, `${client.name} has paid application fee of UGX ${Number(amount).toLocaleString()}. Please confirm payment.`]
            );
        }

        // Notify client
        await dbRun(
            `INSERT INTO notifications (user_id, message, type, created_at) 
            VALUES (?, ?, 'info', datetime('now'))`,
            [clientId, `Your application fee payment of UGX ${Number(amount).toLocaleString()} has been submitted. Awaiting accountant confirmation.`]
        );

        res.status(201).json({
            success: true,
            message: 'Application fee payment submitted successfully! Please wait for accountant confirmation.',
            feeId: result.lastID
        });
    } catch (error) {
        console.error('Error submitting application fee:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Check application fee status
const checkApplicationFeeStatus = async (req, res) => {
    const clientId = req.user.id;
    
    try {
        // Check for confirmed fee
        const confirmedFee = await dbGet(
            `SELECT * FROM loan_fees 
            WHERE user_id = ? 
            AND loan_id IS NULL 
            AND fee_type = 'application'
            AND status = 'confirmed'
            AND payment_confirmed_by IS NOT NULL
            ORDER BY created_at DESC LIMIT 1`,
            [clientId]
        );

        // Check for pending fee
        const pendingFee = await dbGet(
            `SELECT * FROM loan_fees 
            WHERE user_id = ? 
            AND loan_id IS NULL 
            AND fee_type = 'application'
            AND status = 'pending'
            ORDER BY created_at DESC LIMIT 1`,
            [clientId]
        );

        res.json({
            success: true,
            hasConfirmedFee: !!confirmedFee,
            hasPendingFee: !!pendingFee,
            canApplyForLoan: !!confirmedFee,
            confirmedFee: confirmedFee || null,
            pendingFee: pendingFee || null,
            message: confirmedFee 
                ? 'You can now proceed to apply for a loan.' 
                : pendingFee 
                    ? 'Your application fee is pending accountant confirmation.' 
                    : 'Please pay the application fee to proceed.'
        });
    } catch (error) {
        console.error('Error checking application fee status:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Check loan top-up eligibility
const checkTopupEligibility = async (req, res) => {
    const clientId = req.user.id;
    
    try {
        const currentLoan = await dbGet(
            `SELECT * FROM loans 
            WHERE user_id = ? 
            AND status IN ('disbursed', 'active', 'overdue')
            ORDER BY created_at DESC LIMIT 1`,
            [clientId]
        );

        if (!currentLoan) {
            return res.json({
                success: true,
                eligible: false,
                reason: 'No active loan found'
            });
        }

        const amountPaid = currentLoan.amount_paid || 0;
        const totalAmount = currentLoan.amount || 1;
        const repaymentPercentage = (amountPaid / totalAmount) * 100;

        const eligible = repaymentPercentage >= 80;

        // If eligible, check if they have paid application fee for top-up
        let topupFeeStatus = null;
        if (eligible) {
            const confirmedFee = await dbGet(
                `SELECT * FROM loan_fees 
                WHERE user_id = ? 
                AND loan_id IS NULL 
                AND fee_type = 'application'
                AND status = 'confirmed'
                AND payment_confirmed_by IS NOT NULL
                ORDER BY created_at DESC LIMIT 1`,
                [clientId]
            );

            topupFeeStatus = {
                hasPaidFee: !!confirmedFee,
                canProceed: !!confirmedFee
            };
        }

        res.json({
            success: true,
            eligible: eligible,
            repaymentPercentage: repaymentPercentage.toFixed(2),
            amountPaid: amountPaid,
            totalAmount: totalAmount,
            currentLoan: currentLoan,
            topupFeeStatus: topupFeeStatus,
            message: eligible 
                ? (topupFeeStatus?.hasPaidFee 
                    ? 'You are eligible for loan top-up and have paid the application fee!' 
                    : 'You are eligible for loan top-up! Please pay the application fee to proceed.')
                : `You need to pay ${(80 - repaymentPercentage).toFixed(2)}% more to be eligible for top-up.`
        });
    } catch (error) {
        console.error('Error checking top-up eligibility:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

module.exports = {
    getLoanOverview,
    getRepaymentProgress,
    getPortfolioGrowth,
    requestLoanTopUp,
    getNotifications,
    markNotificationRead,
    applyForLoan,
    clearAllNotifications,
    payApplicationFee,
    checkApplicationFeeStatus,
    checkTopupEligibility
};

// Additional client-related functionalities can be added here as needed.