const { dbGet, dbRun } = require('../db');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');

// Ensure JWT_SECRET is available
const JWT_SECRET = process.env.JWT_SECRET || 'your_super_secret_jwt_key_change_this_in_production_12345';

if (!process.env.JWT_SECRET) {
    console.warn('⚠️  authController: JWT_SECRET not found, using fallback');
}

// Login function
exports.login = async (req, res) => {
    const { email, username, password, role } = req.body;
    const identifier = email || username;

    console.log('Login attempt:', { email, username, identifier, role, hasPassword: !!password });

    if (!identifier || !password) {
        console.log('Missing credentials');
        return res.status(400).json({ success: false, error: 'Missing credentials (username/email and password are required).' });
    }

    try {
        // If role is provided, search with role. Otherwise, search without role.
        let user;
        if (role) {
            user = await dbGet(`SELECT * FROM users WHERE (email = ? OR name = ?) AND role = ?`, [identifier, identifier, role]);
        } else {
            user = await dbGet(`SELECT * FROM users WHERE email = ? OR name = ?`, [identifier, identifier]);
        }

        console.log('User found:', user ? `${user.email} (${user.role})` : 'No user found');

        if (!user) {
            console.log('No user found with identifier:', identifier);
            return res.status(401).json({ success: false, error: 'Invalid credentials - user not found.' });
        }

        const passwordMatch = await bcrypt.compare(password, user.password);
        console.log('Password match:', passwordMatch);

        if (passwordMatch) {
            const token = jwt.sign({ id: user.id, role: user.role }, JWT_SECRET, { expiresIn: '1h' });
            console.log('Login successful for:', user.email);
            res.json({ success: true, message: 'Login successful', user: { ...user, password: undefined }, token });
        } else {
            console.log('Password mismatch for:', user.email);
            res.status(401).json({ success: false, error: 'Invalid credentials - wrong password.' });
        }
    } catch (err) {
        console.error('Login error:', err);
        res.status(500).json({ success: false, error: 'Server error: ' + err.message });
    }
};

// Registration function (if needed)
exports.register = async (req, res) => {
    const { username, email, password, role } = req.body;

    if (!username || !email || !password || !role) {
        return res.status(400).json({ error: 'All fields are required.' });
    }

    try {
        const hashedPassword = await bcrypt.hash(password, 10);
        const result = await dbRun(`INSERT INTO users (name, email, password, role) VALUES (?, ?, ?, ?)`, [username, email, hashedPassword, role]);
        res.status(201).json({ message: 'User created successfully', userId: result.lastID });
    } catch (err) {
        console.error('Registration error:', err);
        res.status(500).json({ error: 'Server error' });
    }
};

// Change password function (for logged-in users)
exports.changePassword = async (req, res) => {
    const { currentPassword, newPassword } = req.body;
    const userId = req.user.id; // From authenticate middleware

    if (!currentPassword || !newPassword) {
        return res.status(400).json({ success: false, message: 'Current password and new password are required.' });
    }

    if (newPassword.length < 6) {
        return res.status(400).json({ success: false, message: 'New password must be at least 6 characters long.' });
    }

    try {
        // Get current user
        const user = await dbGet(`SELECT * FROM users WHERE id = ?`, [userId]);
        
        if (!user) {
            return res.status(404).json({ success: false, message: 'User not found.' });
        }

        // Verify current password
        const isValid = await bcrypt.compare(currentPassword, user.password);
        if (!isValid) {
            return res.status(401).json({ success: false, message: 'Current password is incorrect.' });
        }

        // Hash new password
        const hashedPassword = await bcrypt.hash(newPassword, 10);

        // Update password
        await dbRun(`UPDATE users SET password = ? WHERE id = ?`, [hashedPassword, userId]);

        res.json({ success: true, message: 'Password changed successfully' });
    } catch (err) {
        console.error('Change password error:', err);
        res.status(500).json({ success: false, message: 'Server error while changing password' });
    }
};