const { dbAll, dbGet, dbRun } = require('../db');
const PDFDocument = require('pdfkit');
const fs = require('fs');
const path = require('path');

// Helper function to generate unique receipt number
const generateReceiptNumber = async () => {
    try {
        const today = new Date();
        const year = today.getFullYear();
        const month = String(today.getMonth() + 1).padStart(2, '0');
        const day = String(today.getDate()).padStart(2, '0');
        
        // Format: RCT-YYYYMMDD-XXXX
        const prefix = `RCT-${year}${month}${day}`;
        
        // Get count of receipts created today (if repayments table exists)
        try {
            const result = await dbGet(`
                SELECT COUNT(*) as count 
                FROM repayments 
                WHERE receipt_number LIKE ?
            `, [`${prefix}%`]);
            
            const sequence = String((result?.count || 0) + 1).padStart(4, '0');
            return `${prefix}-${sequence}`;
        } catch (tableError) {
            // Repayments table doesn't exist, use simple sequential numbering
            const sequence = String(Math.floor(Math.random() * 9999) + 1).padStart(4, '0');
            return `${prefix}-${sequence}`;
        }
    } catch (error) {
        console.error('Error generating receipt number:', error);
        // Fallback to timestamp-based receipt
        return `RCT-${Date.now()}`;
    }
};

// Helper function to calculate dynamic processing fee based on loan amount
// 50k-199k = 5k, 200k-500k = 15k, 500k+ = 20k
const getProcessingFee = async (loanAmount = 0) => {
    try {
        // Dynamic processing fee based on loan amount tiers
        if (loanAmount >= 500000) {
            return 20000; // 500k and above = 20k
        } else if (loanAmount >= 200000) {
            return 15000; // 200k-500k = 15k
        } else if (loanAmount >= 50000) {
            return 5000;  // 50k-199k = 5k
        } else {
            return 5000;  // Default for amounts below 50k
        }
    } catch (error) {
        console.error('Error calculating processing fee:', error);
        return 5000; // Fallback to default
    }
};

// Helper function to post to general ledger (double-entry)
const postToGeneralLedger = async (entries, description, referenceType, referenceId, postedBy) => {
    try {
        for (const entry of entries) {
            await dbRun(
                `INSERT INTO general_ledger (account_id, debit_amount, credit_amount, description, reference_type, reference_id, posted_by)
                VALUES (?, ?, ?, ?, ?, ?, ?)`,
                [entry.account_id, entry.debit || 0, entry.credit || 0, description, referenceType, referenceId, postedBy]
            );
        }
        
        // Update account balances
        for (const entry of entries) {
            const account = await dbGet(
                'SELECT account_type FROM chart_of_accounts WHERE id = ?',
                [entry.account_id]
            );
            
            if (account) {
                const accountType = account.account_type;
                let balanceChange = 0;
                
                // Assets and Expenses increase with debits, decrease with credits
                if (accountType === 'asset' || accountType === 'expense') {
                    balanceChange = (entry.debit || 0) - (entry.credit || 0);
                }
                // Liabilities, Equity, and Income increase with credits, decrease with debits
                else {
                    balanceChange = (entry.credit || 0) - (entry.debit || 0);
                }
                
                await dbRun(
                    `INSERT INTO account_balances (account_id, current_balance, last_updated)
                    VALUES (?, ?, datetime('now'))
                    ON CONFLICT(account_id) DO UPDATE SET 
                    current_balance = current_balance + ?,
                    last_updated = datetime('now')`,
                    [entry.account_id, balanceChange, balanceChange]
                );
            }
        }
    } catch (error) {
        console.error('Error posting to general ledger:', error);
        throw error;
    }
};

// Get account balances for dashboard
const getAccountBalances = async (req, res) => {
    try {
        const balances = await dbAll(`
            SELECT 
                account_code,
                account_name,
                account_type,
                COALESCE(balance, 0) as balance
            FROM chart_of_accounts
            WHERE account_code IN ('1110', '1120', '1130', '1140', '1210')
            ORDER BY account_code
        `);

        res.json({ success: true, data: balances });
    } catch (error) {
        console.error('Error fetching account balances:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get approved loans ready for disbursement
const getApprovedLoans = async (req, res) => {
    try {
        const loans = await dbAll(`
            SELECT 
                l.id as loan_id,
                l.amount,
                l.amount as requested_amount,
                l.disbursed_amount,
                l.loan_type,
                l.approved_at,
                l.application_fee_status,
                u.name as client_name,
                u.phone as client_phone,
                u.email as client_email,
                o.name as officer_name
            FROM loans l
            INNER JOIN users u ON l.user_id = u.id
            LEFT JOIN users o ON l.officer_id = o.id
            WHERE l.status = 'approved'
            ORDER BY l.approved_at ASC
        `);

        res.json({ success: true, data: loans });
    } catch (error) {
        console.error('Error fetching approved loans:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Disburse loan with accounting entries
const disburseLoanWithAccounting = async (req, res) => {
    const { loanId, disbursement_amount, payment_method, processing_fee_deducted } = req.body;
    const accountantId = req.user.id;  // JWT stores 'id' not 'user_id'

    if (!loanId || !disbursement_amount || !payment_method) {
        return res.status(400).json({ 
            success: false, 
            message: 'Missing required fields' 
        });
    }

    try {
        // Get loan details
        const loanResult = await dbGet(
            'SELECT * FROM loans WHERE id = ? AND status = ?',
            [loanId, 'approved']
        );

        if (!loanResult) {
            return res.status(404).json({ 
                success: false, 
                message: 'Loan not found or not approved' 
            });
        }

        const loanData = loanResult;
        
        // Check if application fee is paid
        if (loanData.application_fee_status !== 'paid' && loanData.application_fee_status !== 'waived') {
            return res.status(400).json({ 
                success: false, 
                message: 'Application fee must be paid before disbursement' 
            });
        }
        
        // Determine which cash/bank account to credit based on payment method
        let cashAccountCode;
        switch(payment_method) {
            case 'cash':
                cashAccountCode = '1110'; // Cash on Hand
                break;
            case 'bank':
                cashAccountCode = '1120'; // Bank Account
                break;
            case 'mobile_money':
                cashAccountCode = '1130'; // Mobile Money
                break;
            default:
                return res.status(400).json({ success: false, message: 'Invalid payment method' });
        }

        // Get account IDs
        const loanReceivableAccount = await dbGet(
            'SELECT id as account_id FROM chart_of_accounts WHERE account_code = ?',
            ['1210'] // Loan Receivables
        );
        
        const cashAccount = await dbGet(
            'SELECT id as account_id FROM chart_of_accounts WHERE account_code = ?',
            [cashAccountCode]
        );

        const feeIncomeAccount = await dbGet(
            'SELECT id as account_id FROM chart_of_accounts WHERE account_code = ?',
            ['4200'] // Fee Income
        ) || await dbGet(
            'SELECT id as account_id FROM chart_of_accounts WHERE account_type = ? LIMIT 1',
            ['income']
        );

        if (!loanReceivableAccount || !cashAccount) {
            return res.status(500).json({ success: false, message: 'Chart of accounts not properly configured' });
        }

        // Get processing fee based on APPROVED amount (disbursed_amount), not requested amount
        const approvedAmount = loanData.disbursed_amount || loanData.amount;
        const systemProcessingFee = await getProcessingFee(approvedAmount);
        const processing_fee = processing_fee_deducted || loanData.processing_fee_amount || systemProcessingFee;
        const net_disbursement = disbursement_amount;

        // Double-entry accounting:
        // Debit: Loan Receivables (full approved amount - Asset increases)
        // Credit: Cash/Bank (net disbursement - Asset decreases)
        // Credit: Fee Income (processing fee - Income increases)
        const entries = [
            {
                account_id: loanReceivableAccount.account_id,
                debit: approvedAmount, // Full APPROVED amount (what client will repay)
                credit: 0
            },
            {
                account_id: cashAccount.account_id,
                debit: 0,
                credit: net_disbursement // What client actually receives
            }
        ];

        // If processing fee was deducted, record it as income
        if (processing_fee > 0 && feeIncomeAccount) {
            entries.push({
                account_id: feeIncomeAccount.account_id,
                debit: 0,
                credit: processing_fee
            });

            // Record processing fee in loan_fees table (status 'paid' means it was collected)
            await dbRun(
                `INSERT INTO loan_fees (loan_id, fee_type, amount, status, paid_amount, payment_date, confirmed_by)
                VALUES (?, 'processing', ?, 'paid', ?, datetime('now'), ?)`,
                [loanId, processing_fee, processing_fee, accountantId]
            );

            // Record processing fee transaction
            await dbRun(
                `INSERT INTO transactions (transaction_type, amount, description, created_by)
                VALUES ('processing_fee', ?, ?, ?)`,
                [
                    processing_fee,
                    `Processing fee deducted from Loan #${loanId} via ${payment_method}`,
                    accountantId
                ]
            );

            // Update loan processing fee status (different constraint: 'pending', 'deducted', 'waived')
            await dbRun(
                'UPDATE loans SET processing_fee_status = \'deducted\' WHERE id = ?',
                [loanId]
            );
        }

        await postToGeneralLedger(
            entries,
            `Loan disbursement to ${loanData.user_id} - Loan #${loanId}`,
            'loan_disbursement',
            loanId,
            accountantId
        );

        // Calculate balance using simple flat rate
        // Balance = Approved Amount × (1 + Interest Rate / 100)
        // Example: 100,000 at 20% = 100,000 × 1.20 = 120,000
        const interest_amount = (approvedAmount * loanData.interest_rate) / 100;
        const totalBalance = parseFloat(approvedAmount) + parseFloat(interest_amount);

        // Calculate installment amount based on repayment mode and duration
        let numberOfPayments;
        const durationMonths = loanData.duration_months || 1;
        
        switch(loanData.repayment_mode) {
            case 'daily':
                // If 1 month = 30 days, total payments = months × 30
                numberOfPayments = durationMonths * 30;
                break;
            case 'weekly':
                // If 1 month = 4 weeks, total payments = months × 4
                numberOfPayments = durationMonths * 4;
                break;
            case 'bi-weekly':
            case 'biweekly':
                // If 1 month = 2 bi-weekly periods, total payments = months × 2
                numberOfPayments = durationMonths * 2;
                break;
            case 'monthly':
            default:
                // Monthly payments = number of months
                numberOfPayments = durationMonths;
                break;
        }
        
        // Installment = Total Balance / Number of Payments
        // Example: 120,000 balance, monthly for 1 month = 120,000 / 1 = 120,000
        // Example: 120,000 balance, weekly for 1 month = 120,000 / 4 = 30,000
        const installmentAmount = totalBalance / numberOfPayments;

        // Calculate next payment date based on repayment mode
        let nextPaymentDate;
        const today = new Date();
        
        switch(loanData.repayment_mode) {
            case 'daily':
                nextPaymentDate = new Date(today.setDate(today.getDate() + 1));
                break;
            case 'weekly':
                nextPaymentDate = new Date(today.setDate(today.getDate() + 7));
                break;
            case 'bi-weekly':
            case 'biweekly':
                nextPaymentDate = new Date(today.setDate(today.getDate() + 14));
                break;
            case 'monthly':
            default:
                nextPaymentDate = new Date(today.setMonth(today.getMonth() + 1));
                break;
        }
        
        const nextPaymentDateStr = nextPaymentDate.toISOString().split('T')[0]; // Format: YYYY-MM-DD

        // Update loan status to disbursed
        // disbursed_amount = full approved amount (what client received)
        // balance = approved amount + interest (total owed)
        // installment_amount = balance divided by number of payments based on repayment mode
        // next_payment_date = calculated based on repayment mode
        await dbRun(
            `UPDATE loans 
            SET status = 'disbursed', 
                disbursed_amount = ?, 
                balance = ?,
                installment_amount = ?,
                next_payment_date = ?,
                disbursed_at = datetime('now'),
                disbursed_by = ?
            WHERE id = ?`,
            [approvedAmount, totalBalance, installmentAmount, nextPaymentDateStr, accountantId, loanId]
        );

        // Record transaction
        await dbRun(
            `INSERT INTO transactions (transaction_type, amount, description, created_by)
            VALUES ('disbursement', ?, ?, ?)`,
            [disbursement_amount, `Loan disbursement for loan #${loanId} via ${payment_method}`, accountantId]
        );

        // Notify client
        await dbRun(
            'INSERT INTO notifications (user_id, message, type) VALUES (?, ?, ?)',
            [loanData.user_id, `Your loan of UGX ${parseFloat(disbursement_amount).toLocaleString()} has been disbursed via ${payment_method}!`, 'approval']
        );

        res.json({ 
            success: true, 
            message: 'Loan disbursed successfully with accounting entries posted' 
        });
    } catch (error) {
        console.error('Error disbursing loan:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Get financial summary
const getFinancialSummary = async (req, res) => {
    try {
        // Calculate total funds, disbursements, deposits, expenditures
        const summary = await dbGet(`
            SELECT 
                COALESCE(SUM(CASE WHEN transaction_type = 'deposit' THEN amount ELSE 0 END), 0) as total_deposits,
                COALESCE(SUM(CASE WHEN transaction_type = 'disbursement' THEN amount ELSE 0 END), 0) as total_disbursements,
                COALESCE(SUM(CASE WHEN transaction_type = 'expenditure' THEN amount ELSE 0 END), 0) as total_expenditures,
                COALESCE(SUM(CASE WHEN transaction_type = 'repayment' THEN amount ELSE 0 END), 0) as total_repayments
            FROM transactions
        `);

        const pettyCashTotal = await dbGet(`
            SELECT COALESCE(SUM(amount), 0) as total FROM petty_cash
        `);

        // Calculate total interest earned from completed loans
        const interestEarned = await dbGet(`
            SELECT 
                COALESCE(SUM((amount * interest_rate * duration_months / 1200)), 0) as total_interest,
                COUNT(*) as completed_loans
            FROM loans
            WHERE status = 'completed'
        `);

        const fundsAvailable = summary.total_deposits + summary.total_repayments 
                             - summary.total_disbursements - summary.total_expenditures;

        res.json({
            success: true,
            data: {
                totalFundsAvailable: fundsAvailable,
                totalDeposits: summary.total_deposits,
                totalDisbursements: summary.total_disbursements,
                totalExpenditures: summary.total_expenditures,
                totalRepayments: summary.total_repayments,
                totalPettyCash: pettyCashTotal.total,
                totalInterestEarned: interestEarned.total_interest,
                completedLoans: interestEarned.completed_loans
            }
        });
    } catch (error) {
        console.error('Error fetching financial summary:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get interest earned from completed loans
const getInterestEarned = async (req, res) => {
    try {
        const { startDate, endDate } = req.query;
        
        let query = `
            SELECT 
                l.id,
                'LN-' || l.id as loan_number,
                u.name as client_name,
                l.amount as principal,
                l.interest_rate,
                l.duration_months,
                (l.amount * l.interest_rate * l.duration_months / 1200) as interest_earned,
                l.amount_paid as total_paid,
                l.disbursed_at,
                l.updated_at as completed_at
            FROM loans l
            JOIN users u ON l.user_id = u.id
            WHERE l.status = 'completed'
        `;
        
        const params = [];
        if (startDate && endDate) {
            query += ' AND l.updated_at BETWEEN ? AND ?';
            params.push(startDate, endDate);
        }
        
        query += ' ORDER BY l.updated_at DESC';
        
        const completedLoans = await dbAll(query, params);
        
        // Calculate totals
        const totals = completedLoans.reduce((acc, loan) => {
            acc.totalPrincipal += parseFloat(loan.principal);
            acc.totalInterest += parseFloat(loan.interest_earned);
            acc.totalCollected += parseFloat(loan.total_paid);
            return acc;
        }, { totalPrincipal: 0, totalInterest: 0, totalCollected: 0 });
        
        res.json({
            success: true,
            data: {
                loans: completedLoans,
                summary: {
                    totalLoans: completedLoans.length,
                    totalPrincipal: totals.totalPrincipal.toFixed(2),
                    totalInterest: totals.totalInterest.toFixed(2),
                    totalCollected: totals.totalCollected.toFixed(2)
                }
            }
        });
    } catch (error) {
        console.error('Error fetching interest earned:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get recent transactions
const getRecentTransactions = async (req, res) => {
    const { limit = 50, type } = req.query;

    try {
        let query = `
            SELECT t.*, u.name as created_by_name
            FROM transactions t
            LEFT JOIN users u ON t.created_by = u.id
        `;

        const params = [];
        if (type) {
            query += ' WHERE transaction_type = ?';
            params.push(type);
        }

        query += ' ORDER BY transaction_date DESC LIMIT ?';
        params.push(parseInt(limit));

        const transactions = await dbAll(query, params);

        res.json({ success: true, data: transactions });
    } catch (error) {
        console.error('Error fetching transactions:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Add new transaction
const addTransaction = async (req, res) => {
    const { transaction_type, amount, description } = req.body;
    const accountantId = req.user.id;

    try {
        if (!['deposit', 'disbursement', 'expenditure'].includes(transaction_type)) {
            return res.status(400).json({ 
                success: false, 
                message: 'Invalid transaction type' 
            });
        }

        await dbRun(
            `INSERT INTO transactions (transaction_type, amount, description, created_by)
            VALUES (?, ?, ?, ?)`,
            [transaction_type, amount, description, accountantId]
        );

        res.json({ success: true, message: 'Transaction added successfully' });
    } catch (error) {
        console.error('Error adding transaction:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get petty cash records and balance
const getPettyCash = async (req, res) => {
    try {
        // Get current balance
        const balance = await dbGet('SELECT * FROM petty_cash_balance ORDER BY id DESC LIMIT 1');
        
        // Get recent transactions
        const transactions = await dbAll(`
            SELECT pc.*, u.name as recorded_by_name
            FROM petty_cash pc
            LEFT JOIN users u ON pc.recorded_by = u.id
            ORDER BY transaction_date DESC
            LIMIT 100
        `);

        res.json({ 
            success: true, 
            data: transactions,
            balance: {
                allocated: balance?.allocated_amount || 0,
                current: balance?.current_balance || 0,
                spent: (balance?.allocated_amount || 0) - (balance?.current_balance || 0)
            }
        });
    } catch (error) {
        console.error('Error fetching petty cash:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Add petty cash record (expense - deducts from balance)
const addPettyCash = async (req, res) => {
    const { amount, description } = req.body;
    const accountantId = req.user.id;

    try {
        // Get current balance
        let balance = await dbGet('SELECT * FROM petty_cash_balance ORDER BY id DESC LIMIT 1');
        
        // If no balance record exists, create one
        if (!balance) {
            await dbRun(
                'INSERT INTO petty_cash_balance (allocated_amount, current_balance, last_updated) VALUES (?, ?, datetime(\'now\'))',
                [0, 0]
            );
            balance = await dbGet('SELECT * FROM petty_cash_balance ORDER BY id DESC LIMIT 1');
        }
        
        if (!balance || balance.current_balance < amount) {
            return res.status(400).json({ 
                success: false, 
                message: `Insufficient petty cash balance. Available: UGX ${(balance?.current_balance || 0).toLocaleString()}` 
            });
        }

        // Record the expense in petty_cash table (negative amount for expense)
        await dbRun(
            'INSERT INTO petty_cash (amount, description, recorded_by) VALUES (?, ?, ?)',
            [-amount, description, accountantId]
        );

        // Deduct from balance
        const newBalance = balance.current_balance - amount;
        await dbRun(
            'UPDATE petty_cash_balance SET current_balance = ?, last_updated = datetime(\'now\') WHERE id = ?',
            [newBalance, balance.id]
        );

        // Record in transactions table as expenditure if it exists
        try {
            await dbRun(
                `INSERT INTO transactions (transaction_type, amount, description, created_by)
                VALUES ('expenditure', ?, ?, ?)`,
                [amount, `Petty Cash: ${description}`, accountantId]
            );
        } catch (err) {
            console.log('Transactions table insert failed:', err.message);
        }

        res.json({ 
            success: true, 
            message: 'Petty cash expense recorded successfully',
            newBalance: newBalance
        });
    } catch (error) {
        console.error('Error adding petty cash:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Allocate petty cash (initial allocation or full reset to target amount)
const allocatePettyCash = async (req, res) => {
    const { amount, description } = req.body;
    const accountantId = req.user.id;

    try {
        if (!amount || amount <= 0) {
            return res.status(400).json({ success: false, message: 'Amount must be greater than 0' });
        }

        // Get current balance
        const balance = await dbGet('SELECT * FROM petty_cash_balance ORDER BY id DESC LIMIT 1');
        
        // Update allocation and set current balance to the new amount
        await dbRun(
            'UPDATE petty_cash_balance SET allocated_amount = ?, current_balance = ?, last_updated = datetime(\'now\'), updated_by = ? WHERE id = ?',
            [amount, amount, accountantId, balance.id]
        );

        // Record allocation in history
        await dbRun(
            'INSERT INTO petty_cash_allocations (amount, allocation_type, description, allocated_by) VALUES (?, ?, ?, ?)',
            [amount, 'allocation', description || 'Petty cash allocation', accountantId]
        );

        // Record in transactions as expenditure (money leaving main account to petty cash)
        await dbRun(
            `INSERT INTO transactions (transaction_type, amount, description, created_by)
            VALUES ('expenditure', ?, ?, ?)`,
            [amount, `Petty Cash Allocation: ${description || 'Initial allocation'}`, accountantId]
        );

        res.json({ 
            success: true, 
            message: 'Petty cash allocated successfully',
            allocated: amount,
            balance: amount
        });
    } catch (error) {
        console.error('Error allocating petty cash:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Top up petty cash (add to current balance to reach allocated amount)
const topUpPettyCash = async (req, res) => {
    const { description } = req.body;
    const accountantId = req.user.id;

    try {
        // Get current balance
        const balance = await dbGet('SELECT * FROM petty_cash_balance ORDER BY id DESC LIMIT 1');
        
        if (!balance || balance.allocated_amount === 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'Please set an allocation amount first' 
            });
        }

        // Calculate top-up amount needed
        const topUpAmount = balance.allocated_amount - balance.current_balance;
        
        if (topUpAmount <= 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'Petty cash is already at full allocation' 
            });
        }

        // Update balance to allocated amount
        await dbRun(
            'UPDATE petty_cash_balance SET current_balance = allocated_amount, last_updated = datetime(\'now\'), updated_by = ? WHERE id = ?',
            [accountantId, balance.id]
        );

        // Record top-up in history
        await dbRun(
            'INSERT INTO petty_cash_allocations (amount, allocation_type, description, allocated_by) VALUES (?, ?, ?, ?)',
            [topUpAmount, 'top-up', description || 'Petty cash top-up', accountantId]
        );

        // Record in transactions as expenditure
        await dbRun(
            `INSERT INTO transactions (transaction_type, amount, description, created_by)
            VALUES ('expenditure', ?, ?, ?)`,
            [topUpAmount, `Petty Cash Top-up: ${description || 'Balance replenishment'}`, accountantId]
        );

        res.json({ 
            success: true, 
            message: `Petty cash topped up with UGX ${topUpAmount.toLocaleString()}`,
            topUpAmount: topUpAmount,
            newBalance: balance.allocated_amount
        });
    } catch (error) {
        console.error('Error topping up petty cash:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get account allocations (Cash, Bank, Mobile Money)
const getAccountAllocations = async (req, res) => {
    try {
        const accounts = await dbAll(`
            SELECT 
                aa.*,
                COALESCE(ca.account_name, 
                    CASE aa.account_code
                        WHEN '1110' THEN 'Cash on Hand'
                        WHEN '1120' THEN 'Bank Account'
                        WHEN '1130' THEN 'Mobile Money'
                        WHEN '1101' THEN 'Cash on Hand'
                        WHEN '1102' THEN 'Cash Reserve'
                        WHEN '1210' THEN 'Loan Receivables'
                        WHEN '1220' THEN 'Interest Receivable'
                        ELSE 'Unknown Account'
                    END
                ) as account_name
            FROM account_allocations aa
            LEFT JOIN chart_of_accounts ca ON aa.account_code = ca.account_code
            ORDER BY aa.account_code ASC
        `);

        res.json({ 
            success: true, 
            data: accounts.map(acc => ({
                ...acc,
                spent: acc.allocated_amount - acc.current_balance
            }))
        });
    } catch (error) {
        console.error('Error fetching account allocations:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Allocate funds to an account (Cash/Bank/Mobile Money)
const allocateAccountFunds = async (req, res) => {
    const { account_code, amount, description } = req.body;
    const accountantId = req.user.id;

    try {
        if (!account_code || !amount || amount <= 0) {
            return res.status(400).json({ success: false, message: 'Account code and valid amount are required' });
        }

        // Get current account from allocations
        const account = await dbGet('SELECT * FROM account_allocations WHERE account_code = ?', [account_code]);
        
        if (!account) {
            return res.status(400).json({ success: false, message: 'Account not found in allocations' });
        }

        // Update allocation and balance in account_allocations table
        await dbRun(
            'UPDATE account_allocations SET allocated_amount = ?, current_balance = ?, last_updated = datetime(\'now\'), updated_by = ? WHERE account_code = ?',
            [amount, amount, accountantId, account_code]
        );

        // Try to record in history (optional - won't fail if table doesn't exist)
        try {
            await dbRun(
                'INSERT INTO account_allocation_history (account_code, amount, transaction_type, description, recorded_by) VALUES (?, ?, ?, ?, ?)',
                [account_code, amount, 'allocation', description || 'Fund allocation', accountantId]
            );
        } catch (historyError) {
            // Silently skip if history table doesn't exist
        }

        res.json({ 
            success: true, 
            message: 'Funds allocated successfully',
            allocated: amount,
            balance: amount
        });
    } catch (error) {
        console.error('Error allocating account funds:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Top up account to allocated amount
const topUpAccount = async (req, res) => {
    const { account_code, description } = req.body;
    const accountantId = req.user.id;

    try {
        if (!account_code) {
            return res.status(400).json({ success: false, message: 'Account code is required' });
        }

        // Get current account
        const account = await dbGet('SELECT * FROM account_allocations WHERE account_code = ?', [account_code]);
        
        if (!account) {
            return res.status(404).json({ success: false, message: 'Account not found' });
        }

        if (account.allocated_amount === 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'Please set an allocation amount first' 
            });
        }

        // Calculate top-up amount
        const topUpAmount = account.allocated_amount - account.current_balance;
        
        if (topUpAmount <= 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'Account is already at full allocation' 
            });
        }

        // Update balance to match allocated amount
        await dbRun(
            'UPDATE account_allocations SET current_balance = allocated_amount, last_updated = datetime(\'now\'), updated_by = ? WHERE account_code = ?',
            [accountantId, account_code]
        );

        // Try to record in history (optional)
        try {
            await dbRun(
                'INSERT INTO account_allocation_history (account_code, amount, transaction_type, description, recorded_by) VALUES (?, ?, ?, ?, ?)',
                [account_code, topUpAmount, 'top-up', description || 'Balance top-up', accountantId]
            );
        } catch (historyError) {
            // Silently skip if history table doesn't exist
        }

        res.json({ 
            success: true, 
            message: `Topped up with UGX ${topUpAmount.toLocaleString()}`,
            topUpAmount: topUpAmount,
            newBalance: account.allocated_amount
        });
    } catch (error) {
        console.error('Error topping up account:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Disburse loan
const disburseLoan = async (req, res) => {
    const { loanId } = req.params;
    const accountantId = req.user.id;

    try {
        // Get loan details
        const loan = await dbGet(
            'SELECT * FROM loans WHERE id = ? AND status = ?',
            [loanId, 'approved']
        );

        if (!loan) {
            return res.status(404).json({ 
                success: false, 
                message: 'Loan not found or not approved' 
            });
        }

        // Update loan status to disbursed
        await dbRun(
            `UPDATE loans 
            SET status = 'disbursed', disbursed_amount = ?, disbursed_at = datetime('now')
            WHERE id = ?`,
            [loan.amount, loanId]
        );

        // Record transaction
        await dbRun(
            `INSERT INTO transactions (transaction_type, amount, description, created_by)
            VALUES ('disbursement', ?, ?, ?)`,
            [loan.amount, `Loan disbursement for loan #${loanId}`, accountantId]
        );

        // Notify client
        await dbRun(
            'INSERT INTO notifications (user_id, message, type) VALUES (?, ?, ?)',
            [loan.user_id, `Your loan of UGX ${loan.amount.toLocaleString()} has been disbursed!`, 'approval']
        );

        res.json({ success: true, message: 'Loan disbursed successfully' });
    } catch (error) {
        console.error('Error disbursing loan:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Generate financial report
const generateReport = async (req, res) => {
    const { start_date, end_date } = req.query;

    try {
        const report = await dbAll(`
            SELECT 
                transaction_type,
                COUNT(*) as count,
                COALESCE(SUM(amount), 0) as total_amount
            FROM transactions
            WHERE transaction_date BETWEEN ? AND ?
            GROUP BY transaction_type
        `, [start_date || '2000-01-01', end_date || new Date().toISOString().split('T')[0]]);

        // Get loan statistics
        const loanStats = await dbGet(`
            SELECT 
                COUNT(*) as total_loans,
                COALESCE(SUM(amount), 0) as total_loaned,
                COALESCE(SUM(amount_paid), 0) as total_collected,
                COALESCE(SUM(balance), 0) as total_outstanding
            FROM loans
            WHERE created_at BETWEEN ? AND ?
        `, [start_date || '2000-01-01', end_date || new Date().toISOString().split('T')[0]]);

        res.json({
            success: true,
            data: {
                transactions: report,
                loanStats: loanStats,
                period: { start_date, end_date }
            }
        });
    } catch (error) {
        console.error('Error generating report:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Generate PDF Report
const generatePDFReport = async (req, res) => {
    const { start_date, end_date } = req.query;

    try {
        // Fetch report data
        const report = await dbAll(`
            SELECT 
                transaction_type,
                COUNT(*) as count,
                COALESCE(SUM(amount), 0) as total_amount
            FROM transactions
            WHERE transaction_date BETWEEN ? AND ?
            GROUP BY transaction_type
        `, [start_date || '2000-01-01', end_date || new Date().toISOString().split('T')[0]]);

        const loanStats = await dbGet(`
            SELECT 
                COUNT(*) as total_loans,
                COALESCE(SUM(amount), 0) as total_loaned,
                COALESCE(SUM(amount_paid), 0) as total_collected,
                COALESCE(SUM(balance), 0) as total_outstanding
            FROM loans
            WHERE created_at BETWEEN ? AND ?
        `, [start_date || '2000-01-01', end_date || new Date().toISOString().split('T')[0]]);

        // Create PDF
        const doc = new PDFDocument({ margin: 50 });
        
        // Set response headers for PDF download
        res.setHeader('Content-Type', 'application/pdf');
        res.setHeader('Content-Disposition', `attachment; filename=financial-report-${Date.now()}.pdf`);
        
        // Pipe PDF to response
        doc.pipe(res);

        // Add institution header
        doc.fontSize(20).font('Helvetica-Bold').text('STARTING POINT FINANCIAL INSTITUTION', { align: 'center' });
        doc.moveDown(0.5);
        doc.fontSize(16).font('Helvetica').text('Financial Report', { align: 'center' });
        doc.moveDown(0.3);
        doc.fontSize(10).text(`Period: ${start_date || 'Inception'} to ${end_date || 'Present'}`, { align: 'center' });
        doc.fontSize(9).text(`Generated: ${new Date().toLocaleString()}`, { align: 'center' });
        doc.moveDown(2);

        // Transaction Summary Section
        doc.fontSize(14).font('Helvetica-Bold').text('Transaction Summary', { underline: true });
        doc.moveDown(1);
        
        if (report && report.length > 0) {
            // Table header
            const tableTop = doc.y;
            const col1 = 50;
            const col2 = 250;
            const col3 = 400;
            
            doc.fontSize(10).font('Helvetica-Bold');
            doc.text('Transaction Type', col1, tableTop);
            doc.text('Count', col2, tableTop);
            doc.text('Total Amount (UGX)', col3, tableTop);
            
            doc.moveTo(col1, tableTop + 15).lineTo(550, tableTop + 15).stroke();
            doc.moveDown(1);
            
            // Table rows
            doc.font('Helvetica').fontSize(10);
            let grandTotal = 0;
            
            report.forEach((row) => {
                const y = doc.y;
                doc.text(row.transaction_type.toUpperCase(), col1, y);
                doc.text(row.count.toString(), col2, y);
                doc.text(Number(row.total_amount).toLocaleString(), col3, y);
                doc.moveDown(0.8);
                grandTotal += Number(row.total_amount);
            });
            
            doc.moveTo(col1, doc.y).lineTo(550, doc.y).stroke();
            doc.moveDown(0.5);
            doc.font('Helvetica-Bold');
            doc.text('TOTAL', col1, doc.y);
            doc.text(grandTotal.toLocaleString(), col3, doc.y);
        } else {
            doc.fontSize(10).font('Helvetica').text('No transactions found for the selected period.');
        }
        
        doc.moveDown(2);

        // Loan Statistics Section
        doc.fontSize(14).font('Helvetica-Bold').text('Loan Portfolio Summary', { underline: true });
        doc.moveDown(1);
        doc.fontSize(10).font('Helvetica');
        
        if (loanStats) {
            doc.text(`Total Loans Issued: ${loanStats.total_loans || 0}`);
            doc.text(`Total Amount Loaned: UGX ${Number(loanStats.total_loaned || 0).toLocaleString()}`);
            doc.text(`Total Amount Collected: UGX ${Number(loanStats.total_collected || 0).toLocaleString()}`);
            doc.text(`Total Outstanding Balance: UGX ${Number(loanStats.total_outstanding || 0).toLocaleString()}`);
            
            const collectionRate = loanStats.total_loaned > 0 
                ? ((loanStats.total_collected / loanStats.total_loaned) * 100).toFixed(2) 
                : 0;
            doc.moveDown(0.5);
            doc.text(`Collection Rate: ${collectionRate}%`);
        }

        // Signature section
        doc.moveDown(3);
        doc.moveTo(50, doc.y).lineTo(550, doc.y).stroke();
        doc.moveDown(2);
        
        const sigY = doc.y;
        doc.text('_____________________', 100, sigY);
        doc.text('_____________________', 350, sigY);
        doc.moveDown(1);
        doc.text('Accountant', 100, doc.y);
        doc.text('Manager', 350, doc.y);
        doc.moveDown(0.5);
        doc.text('Date: _______________', 100, doc.y);
        doc.text('Date: _______________', 350, doc.y);

        // Finalize PDF
        doc.end();
    } catch (error) {
        console.error('Error generating PDF report:', error);
        res.status(500).json({ success: false, message: 'Failed to generate PDF: ' + error.message });
    }
};

// Get notifications
const getNotifications = async (req, res) => {
    const accountantId = req.user.id;

    try {
        // Check for low funds
        const summary = await dbGet(`
            SELECT 
                COALESCE(SUM(CASE WHEN transaction_type = 'deposit' THEN amount ELSE 0 END), 0) as deposits,
                COALESCE(SUM(CASE WHEN transaction_type = 'disbursement' THEN amount ELSE 0 END), 0) as disbursements,
                COALESCE(SUM(CASE WHEN transaction_type = 'expenditure' THEN amount ELSE 0 END), 0) as expenditures,
                COALESCE(SUM(CASE WHEN transaction_type = 'repayment' THEN amount ELSE 0 END), 0) as repayments
            FROM transactions
        `);

        const fundsAvailable = summary.deposits + summary.repayments 
                             - summary.disbursements - summary.expenditures;

        const lowFunds = fundsAvailable < 1000000; // Less than 1M UGX

        // Get pending approvals (approved but not disbursed)
        const pendingDisbursements = await dbGet(
            "SELECT COUNT(*) as count FROM loans WHERE status = 'approved'"
        );

        // Get accountant notifications
        const notifications = await dbAll(
            `SELECT * FROM notifications 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT 20`,
            [accountantId]
        );

        res.json({
            success: true,
            data: {
                lowFunds: lowFunds,
                fundsAvailable: fundsAvailable,
                pendingDisbursements: pendingDisbursements.count,
                notifications: notifications || []
            }
        });
    } catch (error) {
        console.error('Error fetching notifications:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Search for client loans (for payment recording)
const searchClientLoans = async (req, res) => {
    const { search } = req.query;
    
    if (!search) {
        return res.status(400).json({ success: false, message: 'Search term required' });
    }
    
    try {
        const loans = await dbAll(`
            SELECT 
                l.id,
                l.amount as requested_amount,
                l.disbursed_amount,
                l.amount_paid,
                l.balance,
                l.loan_type,
                l.next_payment_date,
                l.status,
                u.name as client_name,
                u.phone as client_phone,
                u.email as client_email
            FROM loans l
            INNER JOIN users u ON l.user_id = u.id
            WHERE l.status IN ('disbursed', 'active', 'overdue')
            AND l.balance > 0
            AND (
                u.name LIKE ? 
                OR u.phone LIKE ? 
                OR u.email LIKE ?
                OR l.id = ?
            )
            ORDER BY u.name
        `, [`%${search}%`, `%${search}%`, `%${search}%`, search]);
        
        res.json({ success: true, data: loans });
    } catch (error) {
        console.error('Error searching client loans:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Record payment for a loan with accounting entries
const recordPaymentWithAccounting = async (req, res) => {
    const { loanId, amount, payment_method, notes } = req.body;
    const accountantId = req.user.id;
    
    if (!loanId || !amount || !payment_method) {
        return res.status(400).json({ success: false, message: 'Missing required fields' });
    }
    
    if (amount <= 0) {
        return res.status(400).json({ success: false, message: 'Payment amount must be greater than 0' });
    }

    // Generate unique receipt number
    const receiptNumber = await generateReceiptNumber();
    
    try {
        // Get loan details
        const loanResult = await dbGet('SELECT * FROM loans WHERE id = ?', [loanId]);
        
        if (!loanResult) {
            return res.status(404).json({ success: false, message: 'Loan not found' });
        }
        
        const loanData = loanResult;
        
        if (!['disbursed', 'active', 'overdue'].includes(loanData.status)) {
            return res.status(400).json({ success: false, message: 'Loan is not active' });
        }
        
        if (parseFloat(amount) > parseFloat(loanData.balance)) {
            return res.status(400).json({ 
                success: false, 
                message: `Payment amount (${amount}) exceeds loan balance (${loanData.balance})` 
            });
        }
        
        // Determine which cash/bank account to debit based on payment method
        let cashAccountCode;
        switch(payment_method) {
            case 'cash':
                cashAccountCode = '1110'; // Cash on Hand
                break;
            case 'bank':
                cashAccountCode = '1120'; // Bank Account
                break;
            case 'mobile_money':
                cashAccountCode = '1130'; // Mobile Money
                break;
            default:
                return res.status(400).json({ success: false, message: 'Invalid payment method' });
        }

        // Get account IDs
        const loanReceivableAccount = await dbGet(
            'SELECT id FROM chart_of_accounts WHERE account_code = ?',
            ['1210'] // Loan Receivables
        );
        
        const cashAccount = await dbGet(
            'SELECT id FROM chart_of_accounts WHERE account_code = ?',
            [cashAccountCode]
        );

        if (!loanReceivableAccount || !cashAccount) {
            return res.status(500).json({ success: false, message: 'Chart of accounts not properly configured' });
        }

        // Double-entry accounting:
        // Debit: Cash/Bank (Asset increases)
        // Credit: Loan Receivables (Asset decreases)
        const entries = [
            {
                account_id: cashAccount.id,
                debit: amount,
                credit: 0
            },
            {
                account_id: loanReceivableAccount.id,
                debit: 0,
                credit: amount
            }
        ];

        await postToGeneralLedger(
            entries,
            `Loan payment received from ${loanData.user_id} - Loan #${loanId}`,
            'loan_repayment',
            loanId,
            accountantId
        );
        
        // Insert payment record with generated receipt number (optional if repayments table exists)
        try {
            await dbRun(
                `INSERT INTO repayments (loan_id, amount, payment_date, payment_method, recorded_by, payment_received_by, receipt_number, notes)
                VALUES (?, ?, datetime('now'), ?, ?, ?, ?, ?)`,
                [loanId, amount, payment_method, accountantId, accountantId, receiptNumber, notes || '']
            );
        } catch (repaymentError) {
            // Repayments table doesn't exist - skip this step (payment still recorded in transactions table)
            console.log('Note: Repayments table not available, skipping detailed payment record');
        }
        
        // Update loan balance and amount paid
        const newAmountPaid = parseFloat(loanData.amount_paid) + parseFloat(amount);
        const newBalance = parseFloat(loanData.balance) - parseFloat(amount);
        const newStatus = newBalance <= 0 ? 'completed' : (loanData.status === 'overdue' ? 'active' : loanData.status);
        
        // Calculate next payment date based on repayment mode (if loan not completed)
        let nextPaymentDate = null;
        if (newBalance > 0) {
            const today = new Date();
            
            switch(loanData.repayment_mode) {
                case 'daily':
                    nextPaymentDate = new Date(today.setDate(today.getDate() + 1));
                    break;
                case 'weekly':
                    nextPaymentDate = new Date(today.setDate(today.getDate() + 7));
                    break;
                case 'monthly':
                    nextPaymentDate = new Date(today.setMonth(today.getMonth() + 1));
                    break;
                default:
                    nextPaymentDate = new Date(today.setMonth(today.getMonth() + 1));
            }
            
            nextPaymentDate = nextPaymentDate.toISOString().split('T')[0]; // Format: YYYY-MM-DD
        }
        
        await dbRun(
            'UPDATE loans SET amount_paid = ?, balance = ?, status = ?, next_payment_date = ? WHERE id = ?',
            [newAmountPaid, newBalance, newStatus, nextPaymentDate, loanId]
        );
        
        // Record transaction
        await dbRun(
            `INSERT INTO transactions (transaction_type, amount, description, created_by)
            VALUES ('repayment', ?, ?, ?)`,
            [amount, `Loan payment - Loan ID: ${loanId} via ${payment_method}`, accountantId]
        );
        
        // Create notification for client
        await dbRun(
            `INSERT INTO notifications (user_id, type, message, created_at)
            VALUES (?, 'payment', ?, datetime('now'))`,
            [loanData.user_id, `Payment of UGX ${Number(amount).toLocaleString()} received via ${payment_method}. New balance: UGX ${Number(newBalance).toLocaleString()}. Receipt: ${receiptNumber}`]
        );
        
        res.json({ 
            success: true, 
            message: 'Payment recorded successfully with accounting entries posted',
            data: {
                newBalance: newBalance,
                newAmountPaid: newAmountPaid,
                loanStatus: newStatus,
                receiptNumber: receiptNumber
            }
        });
    } catch (error) {
        console.error('Error recording payment:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Get all payments for a loan
const getLoanPayments = async (req, res) => {
    const { loanId } = req.params;
    
    try {
        let payments = [];
        
        // Try to get payments from repayments table (if it exists)
        try {
            payments = await dbAll(`
                SELECT 
                    r.id,
                    r.amount,
                    r.payment_date,
                    r.payment_method,
                    r.receipt_number,
                    r.notes,
                    u.name as recorded_by_name
                FROM repayments r
                LEFT JOIN users u ON r.recorded_by = u.id
                WHERE r.loan_id = ?
                ORDER BY r.payment_date DESC
            `, [loanId]);
        } catch (tableError) {
            // Repayments table doesn't exist, return empty array
            console.log('Note: Repayments table not available');
            payments = [];
        }
        
        // Get loan details
        const loanResult = await dbGet(`
            SELECT 
                l.id,
                l.amount as requested_amount,
                l.disbursed_amount,
                l.amount_paid,
                l.balance,
                l.status,
                u.name as client_name,
                u.phone as client_phone
            FROM loans l
            INNER JOIN users u ON l.user_id = u.id
            WHERE l.id = ?
        `, [loanId]);
        
        res.json({ 
            success: true, 
            data: {
                loan: loanResult || null,
                payments: payments
            }
        });
    } catch (error) {
        console.error('Error fetching loan payments:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Edit a payment
const editPayment = async (req, res) => {
    const { paymentId } = req.params;
    const { amount, payment_method, receipt_number, notes } = req.body;
    const accountantId = req.user.id;
    
    if (!amount || !payment_method) {
        return res.status(400).json({ success: false, message: 'Missing required fields' });
    }
    
    try {
        // Check if repayments table exists first
        let oldPayment;
        try {
            oldPayment = await dbGet('SELECT * FROM repayments WHERE id = ?', [paymentId]);
        } catch (tableError) {
            return res.status(404).json({ 
                success: false, 
                message: 'Repayments table not available. Cannot edit payments.' 
            });
        }
        
        if (!oldPayment) {
            return res.status(404).json({ success: false, message: 'Payment not found' });
        }
        
        const oldPaymentData = oldPayment;
        const oldAmount = parseFloat(oldPaymentData.amount);
        const newAmount = parseFloat(amount);
        const amountDifference = newAmount - oldAmount;
        
        // Get loan details
        const loanResult = await dbGet('SELECT * FROM loans WHERE id = ?', [oldPaymentData.loan_id]);
        
        if (!loanResult) {
            return res.status(404).json({ success: false, message: 'Loan not found' });
        }
        
        const loanData = loanResult;
        
        // Calculate new loan balances
        const newAmountPaid = parseFloat(loanData.amount_paid) + amountDifference;
        const newBalance = parseFloat(loanData.balance) - amountDifference;
        
        if (newBalance < 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'Payment amount would result in negative loan balance' 
            });
        }
        
        // Update payment record
        await dbRun(`
            UPDATE repayments 
            SET amount = ?, payment_method = ?, receipt_number = ?, notes = ?
            WHERE id = ?
        `, [amount, payment_method, receipt_number, notes, paymentId]);
        
        // Update loan balances
        const newStatus = newBalance <= 0 ? 'completed' : loanData.status;
        await dbRun(`
            UPDATE loans 
            SET amount_paid = ?, balance = ?, status = ?
            WHERE id = ?
        `, [newAmountPaid, newBalance, oldPaymentData.loan_id]);
        
        // If payment method or amount changed, update accounting entries
        if (oldPaymentData.payment_method !== payment_method || oldAmount !== newAmount) {
            // Reverse old accounting entries
            const oldCashAccountCode = getAccountCodeFromMethod(oldPaymentData.payment_method);
            const oldCashAccount = await dbAll(
                'SELECT id FROM chart_of_accounts WHERE account_code = ?',
                [oldCashAccountCode]
            );
            
            const loanReceivableAccount = await dbAll(
                'SELECT id FROM chart_of_accounts WHERE account_code = ?',
                ['1210']
            );
            
            if (oldCashAccount.length > 0 && loanReceivableAccount.length > 0) {
                // Reverse old entry
                await postToGeneralLedger(
                    [
                        { account_id: oldCashAccount.id, debit: 0, credit: oldAmount },
                        { account_id: loanReceivableAccount.id, debit: oldAmount, credit: 0 }
                    ],
                    `Reversal of payment #${paymentId} (edited)`,
                    'payment_reversal',
                    paymentId,
                    accountantId
                );
                
                // Post new entry
                const newCashAccountCode = getAccountCodeFromMethod(payment_method);
                const newCashAccount = await dbAll(
                    'SELECT id FROM chart_of_accounts WHERE account_code = ?',
                    [newCashAccountCode]
                );
                
                if (newCashAccount) {
                    await postToGeneralLedger(
                        [
                            { account_id: newCashAccount.id, debit: newAmount, credit: 0 },
                            { account_id: loanReceivableAccount.id, debit: 0, credit: newAmount }
                        ],
                        `Updated payment for Loan #${oldPaymentData.loan_id}`,
                        'loan_repayment',
                        oldPaymentData.loan_id,
                        accountantId
                    );
                }
            }
        }
        
        // Notify client
        await dbRun(
            `INSERT INTO notifications (user_id, type, message, created_at)
            VALUES (?, 'payment', ?, datetime('now'))`,
            [loanData.user_id, `Payment updated: UGX ${Number(newAmount).toLocaleString()} via ${payment_method}. New balance: UGX ${Number(newBalance).toLocaleString()}`]
        );
        
        res.json({ 
            success: true, 
            message: 'Payment updated successfully',
            data: {
                newBalance: newBalance,
                newAmountPaid: newAmountPaid,
                loanStatus: newStatus
            }
        });
    } catch (error) {
        console.error('Error editing payment:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Delete a payment
const deletePayment = async (req, res) => {
    const { paymentId } = req.params;
    const accountantId = req.user.id;
    
    try {
        // Check if repayments table exists first
        let payment;
        try {
            payment = await dbGet('SELECT * FROM repayments WHERE id = ?', [paymentId]);
        } catch (tableError) {
            return res.status(404).json({ 
                success: false, 
                message: 'Repayments table not available. Cannot delete payments.' 
            });
        }
        
        if (!payment) {
            return res.status(404).json({ success: false, message: 'Payment not found' });
        }
        
        const paymentData = payment;
        const amount = parseFloat(paymentData.amount);
        
        // Get loan details
        const loanResult = await dbGet('SELECT * FROM loans WHERE id = ?', [paymentData.loan_id]);
        
        if (!loanResult) {
            return res.status(404).json({ success: false, message: 'Loan not found' });
        }
        
        const loanData = loanResult;
        
        // Calculate new loan balances (reverse the payment)
        const newAmountPaid = parseFloat(loanData.amount_paid) - amount;
        const newBalance = parseFloat(loanData.balance) + amount;
        
        // Delete payment record
        await dbRun('DELETE FROM repayments WHERE id = ?', [paymentId]);
        
        // Update loan balances
        const newStatus = newBalance > 0 ? 'active' : loanData.status;
        await dbRun(`
            UPDATE loans 
            SET amount_paid = ?, balance = ?, status = ?
            WHERE id = ?
        `, [newAmountPaid, newBalance, paymentData.loan_id]);
        
        // Reverse accounting entries
        const cashAccountCode = getAccountCodeFromMethod(paymentData.payment_method);
        const cashAccount = await dbAll(
            'SELECT id FROM chart_of_accounts WHERE account_code = ?',
            [cashAccountCode]
        );
        
        const loanReceivableAccount = await dbAll(
            'SELECT id FROM chart_of_accounts WHERE account_code = ?',
            ['1210']
        );
        
        if (cashAccount.length > 0 && loanReceivableAccount.length > 0) {
            await postToGeneralLedger(
                [
                    { account_id: cashAccount.id, debit: 0, credit: amount },
                    { account_id: loanReceivableAccount.id, debit: amount, credit: 0 }
                ],
                `Reversal of deleted payment #${paymentId} for Loan #${paymentData.loan_id}`,
                'payment_deletion',
                paymentId,
                accountantId
            );
        }
        
        // Notify client
        await dbRun(
            `INSERT INTO notifications (user_id, type, message, created_at)
            VALUES (?, 'alert', ?, datetime('now'))`,
            [loanData.user_id, `A payment of UGX ${Number(amount).toLocaleString()} was removed. New balance: UGX ${Number(newBalance).toLocaleString()}`]
        );
        
        res.json({ 
            success: true, 
            message: 'Payment deleted successfully',
            data: {
                newBalance: newBalance,
                newAmountPaid: newAmountPaid,
                loanStatus: newStatus
            }
        });
    } catch (error) {
        console.error('Error deleting payment:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// Helper function to get account code from payment method
function getAccountCodeFromMethod(method) {
    const methodLower = (method || '').toLowerCase();
    if (methodLower === 'cash') return '1110';
    if (methodLower === 'bank') return '1120';
    if (methodLower === 'mobile_money' || methodLower === 'mobile money') return '1130';
    return '1110'; // Default to cash
}

// Check for overdue loans and send alerts
const checkOverdueLoans = async (req, res) => {
    try {
        // Get all active/disbursed loans with upcoming or overdue payments
        const loans = await dbAll(`
            SELECT 
                l.id,
                l.user_id,
                l.officer_id,
                l.amount,
                l.balance,
                l.next_payment_date,
                l.repayment_mode,
                l.duration_months,
                l.status,
                l.disbursed_at,
                u.name as client_name,
                u.phone as client_phone,
                o.name as officer_name,
                CAST(julianday('now') - julianday(l.next_payment_date) AS INTEGER) as days_overdue
            FROM loans l
            INNER JOIN users u ON l.user_id = u.id
            LEFT JOIN users o ON l.officer_id = o.id
            WHERE l.status IN ('disbursed', 'active', 'overdue')
            AND l.balance > 0
            AND l.next_payment_date IS NOT NULL
            ORDER BY l.next_payment_date ASC
        `);
        
        const alerts = {
            overdue: [],
            dueSoon: [],
            upcomingThisWeek: []
        };
        
        for (const loan of loans) {
            const daysOverdue = loan.days_overdue || 0;
            
            if (daysOverdue > 0) {
                // Overdue
                alerts.overdue.push({
                    ...loan,
                    priority: daysOverdue > 7 ? 'high' : daysOverdue > 3 ? 'medium' : 'low'
                });
                
                // Update loan status to overdue if not already
                if (loan.status !== 'overdue') {
                    await dbRun('UPDATE loans SET status = ? WHERE id = ?', ['overdue', loan.id]);
                    
                    // Notify officer
                    if (loan.officer_id) {
                        await dbRun(
                            `INSERT INTO notifications (user_id, type, message, created_at)
                            VALUES (?, 'overdue', ?, datetime('now'))`,
                            [loan.officer_id, `⚠️ OVERDUE: ${loan.client_name} - Loan #${loan.id} is ${daysOverdue} day(s) overdue. Balance: UGX ${Number(loan.balance).toLocaleString()}`]
                        );
                    }
                    
                    // Notify client
                    await dbRun(
                        `INSERT INTO notifications (user_id, type, message, created_at)
                        VALUES (?, 'overdue', ?, datetime('now'))`,
                        [loan.user_id, `⚠️ Your loan payment is ${daysOverdue} day(s) overdue. Please make a payment. Balance: UGX ${Number(loan.balance).toLocaleString()}`]
                    );
                }
            } else if (daysOverdue >= -1) {
                // Due today or tomorrow
                alerts.dueSoon.push(loan);
                
                // Notify officer
                if (loan.officer_id) {
                    await dbRun(
                        `INSERT INTO notifications (user_id, type, message, created_at)
                        VALUES (?, 'alert', ?, datetime('now'))`,
                        [loan.officer_id, `📅 Payment due ${daysOverdue === 0 ? 'TODAY' : 'TOMORROW'}: ${loan.client_name} - Loan #${loan.id}. Balance: UGX ${Number(loan.balance).toLocaleString()}`]
                    );
                }
                
                // Notify client
                await dbRun(
                    `INSERT INTO notifications (user_id, type, message, created_at)
                    VALUES (?, 'alert', ?, datetime('now'))`,
                    [loan.user_id, `📅 Reminder: Your loan payment is due ${daysOverdue === 0 ? 'TODAY' : 'TOMORROW'}. Balance: UGX ${Number(loan.balance).toLocaleString()}`]
                );
            } else if (daysOverdue >= -7) {
                // Due within next 7 days
                alerts.upcomingThisWeek.push(loan);
            }
        }
        
        res.json({ 
            success: true, 
            message: 'Loan alerts checked and notifications sent',
            data: alerts
        });
    } catch (error) {
        console.error('Error checking overdue loans:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

// ============================================
// APPLICATION FEE CONFIRMATION (ACCOUNTANT)
// ============================================

// Get pending application fees for accountant to confirm
const getPendingApplicationFees = async (req, res) => {
    try {
        const fees = await dbAll(
            `SELECT 
                lf.id as fee_id,
                lf.user_id,
                lf.transaction_reference,
                lf.fee_type,
                lf.amount,
                lf.payment_method,
                lf.created_at,
                u.name as client_name,
                u.phone as client_phone,
                u.email as client_email
            FROM loan_fees lf
            JOIN users u ON lf.user_id = u.id
            WHERE lf.fee_type = 'application'
            AND lf.status = 'pending'
            ORDER BY lf.created_at DESC`
        );

        res.json({ success: true, data: fees });
    } catch (error) {
        console.error('Error fetching pending fees:', error);
        res.status(500).json({ success: false, message: error.message });
    }
};

// Confirm application fee payment
const confirmApplicationFee = async (req, res) => {
    try {
        const { fee_id } = req.body;
        const accountant_id = req.user.id;

        // Get the fee record
        const fee = await dbGet(
            'SELECT * FROM loan_fees WHERE id = ? AND fee_type = ?',
            [fee_id, 'application']
        );

        if (!fee) {
            return res.status(404).json({ success: false, message: 'Fee record not found' });
        }

        if (fee.status === 'confirmed') {
            return res.status(400).json({ success: false, message: 'Fee already confirmed' });
        }

        // Update the fee status
        await dbRun(
            `UPDATE loan_fees 
            SET status = 'confirmed', 
                payment_confirmed_by = ?, 
                payment_confirmed_at = datetime('now')
            WHERE id = ?`,
            [accountant_id, fee_id]
        );

        // Record transaction in transactions table
        await dbRun(
            `INSERT INTO transactions (transaction_type, amount, description, created_by)
            VALUES ('application_fee', ?, ?, ?)`,
            [
                fee.amount, 
                `Application fee confirmed - Fee #${fee_id} (Ref: ${fee.transaction_reference || 'N/A'}) via ${fee.payment_method}`,
                accountant_id
            ]
        );

        // Post to general ledger: Cash (Debit), Application Fee Income (Credit)
        const cashAccount = await dbGet(
            'SELECT id FROM chart_of_accounts WHERE account_code = ?',
            ['1110'] // Cash account
        );
        
        const feeIncomeAccount = await dbGet(
            'SELECT id FROM chart_of_accounts WHERE account_code = ?',
            ['4120'] // Application Fee Income
        );

        if (cashAccount && feeIncomeAccount) {
            await postToGeneralLedger(
                [
                    { account_id: cashAccount.id, debit: fee.amount, credit: 0 },
                    { account_id: feeIncomeAccount.id, debit: 0, credit: fee.amount }
                ],
                `Application fee confirmed (Fee #${fee_id}, Ref: ${fee.transaction_reference || 'N/A'})`,
                'fee',
                fee_id,
                accountant_id
            );
        }

        // Check if client has an approved loan waiting for application fee
        const approvedLoan = await dbGet(
            `SELECT id FROM loans 
            WHERE user_id = ? 
            AND status = 'approved' 
            AND (application_fee_status IS NULL OR application_fee_status = 'pending' OR application_fee_status = '')
            ORDER BY approved_at DESC LIMIT 1`,
            [fee.user_id]
        );

        // If there's an approved loan waiting, update its application_fee_status to 'paid'
        if (approvedLoan) {
            await dbRun(
                `UPDATE loans 
                SET application_fee_status = 'paid'
                WHERE id = ?`,
                [approvedLoan.id]
            );

            // Also link this fee to the loan
            await dbRun(
                `UPDATE loan_fees 
                SET loan_id = ?
                WHERE id = ?`,
                [approvedLoan.id, fee_id]
            );
        }

        // Notify client
        await dbRun(
            `INSERT INTO notifications (user_id, message, type) 
            VALUES (?, ?, 'fee_confirmed')`,
            [fee.user_id, `Your application fee of UGX ${fee.amount.toLocaleString()} has been confirmed. You can now apply for a loan.`]
        );

        res.json({
            success: true,
            message: 'Application fee confirmed successfully'
        });
    } catch (error) {
        console.error('Error confirming application fee:', error);
        res.status(500).json({ success: false, message: error.message });
    }
};

// Calculate disbursement amount (deduct processing fee)
const calculateDisbursementAmount = async (req, res) => {
    try {
        const { loan_id } = req.params;

        const loan = await dbGet(
            `SELECT 
                l.*,
                u.name as client_name
            FROM loans l
            JOIN users u ON l.user_id = u.id
            WHERE l.id = ?`,
            [loan_id]
        );

        if (!loan) {
            return res.status(404).json({ success: false, message: 'Loan not found' });
        }

        // Use APPROVED amount (disbursed_amount) not requested amount for calculation
        const approvedAmount = loan.disbursed_amount || loan.amount;
        
        // Calculate dynamic processing fee based on APPROVED amount
        const processing_fee = await getProcessingFee(approvedAmount);
        const net_disbursement = approvedAmount - processing_fee;

        res.json({
            success: true,
            data: {
                loan_id: loan.loan_id,
                client_name: loan.client_name,
                approved_amount: approvedAmount,
                processing_fee: processing_fee,
                net_disbursement: net_disbursement,
                client_repays: approvedAmount, // Client pays back the full APPROVED amount
                processing_fee_status: loan.processing_fee_status
            }
        });
    } catch (error) {
        console.error('Error calculating disbursement:', error);
        res.status(500).json({ success: false, message: error.message });
    }
};

// ============================================
// DELETE OPERATIONS
// ============================================

// Delete transaction
const deleteTransaction = async (req, res) => {
    try {
        const { transactionId } = req.params;
        
        // Check if transaction exists
        const transaction = await dbGet('SELECT * FROM transactions WHERE id = ?', [transactionId]);
        if (!transaction) {
            return res.status(404).json({ success: false, message: 'Transaction not found' });
        }
        
        // Delete transaction
        await dbRun('DELETE FROM transactions WHERE id = ?', [transactionId]);
        
        res.json({ success: true, message: 'Transaction deleted successfully' });
    } catch (error) {
        console.error('Error deleting transaction:', error);
        res.status(500).json({ success: false, message: 'Failed to delete transaction' });
    }
};

// Delete petty cash transaction
const deletePettyCashTransaction = async (req, res) => {
    try {
        const { transactionId } = req.params;
        
        // Get transaction details
        const transaction = await dbGet('SELECT * FROM petty_cash WHERE id = ?', [transactionId]);
        if (!transaction) {
            return res.status(404).json({ success: false, message: 'Petty cash transaction not found' });
        }
        
        // Update petty cash balance
        const pettyCashBalance = await dbGet('SELECT * FROM petty_cash_balance WHERE id = 1');
        
        if (transaction.transaction_type === 'allocation' || transaction.transaction_type === 'top-up') {
            // If deleting allocation/top-up, decrease both allocated and current balance
            await dbRun(
                'UPDATE petty_cash_balance SET allocated_amount = allocated_amount - ?, current_balance = current_balance - ? WHERE id = 1',
                [transaction.amount, transaction.amount]
            );
        } else {
            // If deleting expense, increase current balance (refund)
            await dbRun(
                'UPDATE petty_cash_balance SET current_balance = current_balance + ? WHERE id = 1',
                [transaction.amount]
            );
        }
        
        // Delete transaction
        await dbRun('DELETE FROM petty_cash WHERE id = ?', [transactionId]);
        
        res.json({ success: true, message: 'Petty cash transaction deleted successfully' });
    } catch (error) {
        console.error('Error deleting petty cash transaction:', error);
        res.status(500).json({ success: false, message: 'Failed to delete petty cash transaction' });
    }
};

// Clear all notifications
const clearAllNotifications = async (req, res) => {
    try {
        const userId = req.user.userId;
        await dbRun('DELETE FROM notifications WHERE user_id = ?', [userId]);
        res.json({ success: true, message: 'All notifications cleared' });
    } catch (error) {
        console.error('Error clearing notifications:', error);
        res.status(500).json({ success: false, message: 'Failed to clear notifications' });
    }
};

// =============== SALARY MANAGEMENT ===============

// Get staff list (all users except clients)
const getStaffList = async (req, res) => {
    try {
        const staff = await dbAll(
            `SELECT id, name, email, role 
            FROM users 
            WHERE role IN ('manager', 'officer', 'accountant')
            ORDER BY role, name`
        );

        res.json({ success: true, data: staff });
    } catch (error) {
        console.error('Error fetching staff list:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Get salary payment history
const getSalaryHistory = async (req, res) => {
    try {
        const salaries = await dbAll(
            `SELECT 
                s.*,
                u.name as staff_name,
                u.role as staff_role,
                a.name as paid_by_name
            FROM salaries s
            JOIN users u ON s.staff_id = u.id
            LEFT JOIN users a ON s.paid_by = a.id
            ORDER BY s.payment_date DESC
            LIMIT 100`
        );

        res.json({ success: true, data: salaries });
    } catch (error) {
        console.error('Error fetching salary history:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Pay staff salary
const paySalary = async (req, res) => {
    const {
        staff_id,
        salary_period,
        base_salary,
        bonuses = 0,
        deductions = 0,
        payment_method,
        reference,
        notes
    } = req.body;

    const accountantId = req.user.id;

    if (!staff_id || !salary_period || !base_salary || !payment_method) {
        return res.status(400).json({
            success: false,
            message: 'Missing required fields'
        });
    }

    const net_amount = parseFloat(base_salary) + parseFloat(bonuses) - parseFloat(deductions);

    if (net_amount <= 0) {
        return res.status(400).json({
            success: false,
            message: 'Net salary amount must be greater than zero'
        });
    }

    try {
        // Check if salary already paid for this period
        const existing = await dbGet(
            'SELECT id FROM salaries WHERE staff_id = ? AND salary_period = ?',
            [staff_id, salary_period]
        );

        if (existing) {
            return res.status(400).json({
                success: false,
                message: 'Salary for this period has already been paid'
            });
        }

        // Get staff details
        const staff = await dbGet('SELECT name, role FROM users WHERE id = ?', [staff_id]);
        
        if (!staff) {
            return res.status(404).json({
                success: false,
                message: 'Staff member not found'
            });
        }

        // Insert salary payment
        await dbRun(
            `INSERT INTO salaries (
                staff_id, salary_period, base_salary, bonuses, deductions,
                payment_method, reference, notes, payment_date, paid_by
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, datetime('now'), ?)`,
            [
                staff_id, salary_period, base_salary, bonuses, deductions,
                payment_method, reference, notes, accountantId
            ]
        );

        // Get account IDs for double-entry bookkeeping
        const salaryExpenseAccount = await dbGet(
            'SELECT id as account_id FROM chart_of_accounts WHERE account_code = ?',
            ['6100'] // Salary Expense
        );

        const cashAccount = await dbGet(
            'SELECT id as account_id FROM chart_of_accounts WHERE account_code = ?',
            [payment_method === 'Cash' ? '1110' : payment_method === 'Bank Transfer' ? '1120' : '1130']
        );

        if (salaryExpenseAccount && cashAccount) {
            // Post to general ledger: Debit Salary Expense, Credit Cash/Bank
            const glDescription = `Salary payment to ${staff.name} (${staff.role}) for ${salary_period}`;
            
            await dbRun(
                `INSERT INTO general_ledger (
                    account_id, transaction_date, description, debit, credit, reference
                ) VALUES (?, datetime('now'), ?, ?, 0, ?)`,
                [salaryExpenseAccount.account_id, glDescription, net_amount, reference || 'SAL']
            );

            await dbRun(
                `INSERT INTO general_ledger (
                    account_id, transaction_date, description, debit, credit, reference
                ) VALUES (?, datetime('now'), ?, 0, ?, ?)`,
                [cashAccount.account_id, glDescription, net_amount, reference || 'SAL']
            );
        }

        // Notify the staff member
        await dbRun(
            `INSERT INTO notifications (user_id, message, type, created_at) 
            VALUES (?, ?, 'info', datetime('now'))`,
            [staff_id, `Your salary for ${salary_period} (UGX ${net_amount.toLocaleString()}) has been processed.`]
        );

        res.json({
            success: true,
            message: 'Salary paid successfully'
        });

    } catch (error) {
        console.error('Error paying salary:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
};

// Update exports
// Search payment by receipt number
const searchPaymentByReceipt = async (req, res) => {
    try {
        const { receiptNumber } = req.params;

        let payment;
        try {
            payment = await dbGet(`
                SELECT 
                    r.*,
                    l.user_id,
                    l.loan_type,
                    l.amount as loan_amount,
                    u.name as client_name,
                    u.phone as client_phone,
                    u.email as client_email,
                    rec.name as recorded_by_name
                FROM repayments r
                INNER JOIN loans l ON r.loan_id = l.id
                INNER JOIN users u ON l.user_id = u.id
                LEFT JOIN users rec ON r.recorded_by = rec.id
                WHERE r.receipt_number = ?
            `, [receiptNumber]);
        } catch (tableError) {
            return res.status(404).json({ 
                success: false, 
                message: 'Repayments table not available. Cannot search payments.' 
            });
        }

        if (!payment) {
            return res.status(404).json({ 
                success: false, 
                message: 'Payment not found with this receipt number' 
            });
        }

        res.json({ 
            success: true, 
            payment 
        });
    } catch (error) {
        console.error('Error searching payment:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Failed to search payment' 
        });
    }
};

// Generate payment receipt PDF
const generatePaymentReceipt = async (req, res) => {
    try {
        const { receiptNumber } = req.params;

        let payment;
        try {
            payment = await dbGet(`
                SELECT 
                    r.*,
                    l.user_id,
                    l.loan_type,
                    l.amount as loan_amount,
                    l.balance as current_balance,
                    u.name as client_name,
                    u.phone as client_phone,
                    u.email as client_email,
                    u.village,
                    u.parish,
                    rec.name as recorded_by_name
                FROM repayments r
                INNER JOIN loans l ON r.loan_id = l.id
                INNER JOIN users u ON l.user_id = u.id
                LEFT JOIN users rec ON r.recorded_by = rec.id
                WHERE r.receipt_number = ?
            `, [receiptNumber]);
        } catch (tableError) {
            return res.status(404).json({ 
                success: false, 
                message: 'Repayments table not available. Cannot generate receipt.' 
            });
        }

        if (!payment) {
            return res.status(404).json({ 
                success: false, 
                message: 'Payment not found' 
            });
        }

        // Create PDF
        const doc = new PDFDocument({ margin: 50 });
        
        // Set response headers
        res.setHeader('Content-Type', 'application/pdf');
        res.setHeader('Content-Disposition', `inline; filename=receipt-${receiptNumber}.pdf`);
        
        doc.pipe(res);

        // Header
        doc.fontSize(20).font('Helvetica-Bold').text('SPFI LOAN MANAGEMENT SYSTEM', { align: 'center' });
        doc.fontSize(16).text('PAYMENT RECEIPT', { align: 'center' });
        doc.moveDown();
        doc.fontSize(10).font('Helvetica').text('Starting Point Financial Institution', { align: 'center' });
        doc.text('P.O. Box 123, Kampala, Uganda', { align: 'center' });
        doc.text('Tel: +256 XXX XXXXXX | Email: info@spfi.com', { align: 'center' });
        doc.moveDown(2);

        // Receipt Info
        doc.fontSize(12).font('Helvetica-Bold').text(`Receipt Number: ${payment.receipt_number}`, { underline: true });
        doc.font('Helvetica').text(`Date: ${new Date(payment.payment_date).toLocaleDateString('en-GB')}`);
        doc.text(`Time: ${new Date(payment.payment_date).toLocaleTimeString('en-GB')}`);
        doc.moveDown(1.5);

        // Client Info
        doc.fontSize(11).font('Helvetica-Bold').text('RECEIVED FROM:');
        doc.font('Helvetica');
        doc.text(`Name: ${payment.client_name}`);
        doc.text(`Phone: ${payment.client_phone}`);
        if (payment.client_email) doc.text(`Email: ${payment.client_email}`);
        if (payment.village) doc.text(`Location: ${payment.village}${payment.parish ? ', ' + payment.parish : ''}`);
        doc.moveDown(1.5);

        // Payment Details
        doc.fontSize(11).font('Helvetica-Bold').text('PAYMENT DETAILS:');
        doc.font('Helvetica');
        doc.text(`Loan Type: ${payment.loan_type}`);
        doc.text(`Loan ID: #${payment.loan_id}`);
        doc.text(`Payment Method: ${payment.payment_method.replace('_', ' ').toUpperCase()}`);
        doc.moveDown(1);

        // Amount Box
        doc.rect(50, doc.y, 500, 80).stroke();
        doc.moveDown(0.5);
        doc.fontSize(12).font('Helvetica-Bold').text('AMOUNT PAID:', 60);
        doc.fontSize(20).text(`UGX ${Number(payment.amount).toLocaleString()}`, 60, doc.y, { underline: true });
        doc.moveDown(2);

        // Loan Status
        doc.fontSize(11).font('Helvetica');
        doc.text(`Current Loan Balance: UGX ${Number(payment.current_balance).toLocaleString()}`);
        doc.moveDown(1.5);

        // Notes
        if (payment.notes) {
            doc.fontSize(10).font('Helvetica-Bold').text('Notes:');
            doc.font('Helvetica').text(payment.notes, { width: 500 });
            doc.moveDown(1.5);
        }

        // Footer
        doc.moveDown(2);
        doc.fontSize(10).font('Helvetica-Bold').text(`Received by: ${payment.recorded_by_name || 'System'}`);
        doc.moveDown(2);
        doc.fontSize(8).font('Helvetica-Oblique').text('This is a computer-generated receipt and does not require a signature.', { align: 'center' });
        doc.text('Thank you for your payment!', { align: 'center' });
        doc.moveDown();
        doc.text('Developed by CodToInnovate Africa', { align: 'center' });

        doc.end();
    } catch (error) {
        console.error('Error generating receipt:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Failed to generate receipt' 
        });
    }
};

// Add funds to account (capital injection/deposit)
const addFunds = async (req, res) => {
    const { account_code, amount, source, description } = req.body;
    const accountantId = req.user.id;

    try {
        if (!account_code || !amount || amount <= 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'Account code and valid amount are required' 
            });
        }

        // Verify account exists
        const account = await dbGet(
            'SELECT * FROM chart_of_accounts WHERE account_code = ?',
            [account_code]
        );

        if (!account) {
            return res.status(404).json({ 
                success: false, 
                message: 'Account not found' 
            });
        }

        // Record the deposit transaction (account_code included in description since table doesn't have that column)
        const fullDescription = `${account_code} - Capital Injection - ${source}: ${description}`;
        
        await dbRun(
            `INSERT INTO transactions (transaction_type, amount, description, created_by)
            VALUES ('deposit', ?, ?, ?)`,
            [amount, fullDescription, accountantId]
        );

        // Update account balance in chart_of_accounts
        await dbRun(
            `UPDATE chart_of_accounts 
             SET balance = balance + ? 
             WHERE account_code = ?`,
            [amount, account_code]
        );

        res.json({ 
            success: true, 
            message: 'Funds added successfully',
            amount: amount,
            account: account_code
        });
    } catch (error) {
        console.error('Error adding funds:', error);
        res.status(500).json({ success: false, message: 'Server error: ' + error.message });
    }
};

module.exports = {
    getFinancialSummary,
    getRecentTransactions,
    addTransaction,
    // Petty cash
    getPettyCash,
    addPettyCash,
    allocatePettyCash,
    topUpPettyCash,
    // Add funds
    addFunds,
    // Account allocations
    getAccountAllocations,
    allocateAccountFunds,
    topUpAccount,
    // Disbursements
    disburseLoan,
    generateReport,
    getNotifications,
    searchClientLoans,
    recordPayment: recordPaymentWithAccounting,
    // New accounting methods
    getAccountBalances,
    getApprovedLoans,
    disburseLoanWithAccounting,
    recordPaymentWithAccounting,
    // Payment management
    getLoanPayments,
    editPayment,
    deletePayment,
    // Alerts and follow-ups
    checkOverdueLoans,
    // PDF Report
    generatePDFReport,
    // Interest tracking
    getInterestEarned,
    // Application fee confirmation
    getPendingApplicationFees,
    confirmApplicationFee,
    // Loan fees
    calculateDisbursementAmount,
    // Delete operations
    deleteTransaction,
    deletePettyCashTransaction,
    clearAllNotifications,
    // Salary management
    getStaffList,
    getSalaryHistory,
    paySalary,
    // Receipt management
    searchPaymentByReceipt,
    generatePaymentReceipt
};
